
/*--------------------------------------------------------------------*/
/*--- A signal handler checker                           cr_main.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Crocus, a tool for checking signal handlers.

   Copyright (C) 2000-2003 Julian Seward 
      jseward@acm.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "vg_skin.h"

VG_DETERMINE_INTERFACE_VERSION

#include <errno.h>

// XXX: could detect longjmp out of a handler by doing clever things with
//      noticing indirect jumps and how %esp changes?  Would require
//      extra instrumentation, though.

// XXX: could artificially send lots of extra signals, to stress test the
//      handling?

// XXX: both should be stacks, to cope with signal handlers being
//      interrupted by other signals
//
// 0 if none, -sigNum if signal delivered but handler not called yet,
// otherwise sigNum
static Int in_signal_handler = 0;
static Int old_errno = 0;

/*--------------------------------------------------------------------*/
/*--- Command line options                                         ---*/
/*--------------------------------------------------------------------*/

/* Be strict about which functions are considered reentrant?  Default: NO */
static Bool clo_strict = False;

/* Trace all function calls within a handler?  Default: NO */
static Bool clo_trace_handlers = False;

Bool SK_(process_cmd_line_option)(Char* arg)
{
   VG_(printf)("arg = `%s`\n\n", arg);

   /* normal use */
   if      (VG_CLO_STREQ(arg, "--strict=yes"))
      clo_strict = True;
   else if (VG_CLO_STREQ(arg, "--strict=no"))
      clo_strict = False;

   /* debugging */
   else if (VG_CLO_STREQ(arg, "--trace-handlers=yes"))
      clo_trace_handlers = True;
   else if (VG_CLO_STREQ(arg, "--trace-handlers=no"))
      clo_trace_handlers = False;
   
   else
      return False;

   return True;
}

void SK_(print_usage)(void)
{
   VG_(printf)(
"    --strict=no|yes           be very strict on reentrancy? [no]\n"
   );
}

void SK_(print_debug_usage)(void)
{
   VG_(printf)(
"    --trace-handlers=no|yes   show all calls within handlers? [no]\n"
   );
}

/*--------------------------------------------------------------------*/
/*--- Error handling                                               ---*/
/*--------------------------------------------------------------------*/

static void non_reentrant_call_error(Char* fnname, Int sigNum)
{
   VG_(message)(Vg_UserMsg,
                "signal %d handler (%s), non-reentrant call: %s",
                sigNum, VG_(strsignal)(sigNum), fnname);

   VG_(pp_ExeContext)( VG_(get_ExeContext)(NULL) );
}

static void errno_changed_error(ThreadId tid, Int sigNum, Int old,
                                Int new)
{
   VG_(message)(Vg_UserMsg, 
                "signal %d handler (%s), changed `errno' (%d --> %d)",
                 sigNum, VG_(strsignal)(sigNum), old, new);

   VG_(pp_ExeContext)( VG_(get_ExeContext)( VG_(get_ThreadState)(tid) ) );
}

/*--------------------------------------------------------------------*/
/*--- Runtime checking                                             ---*/
/*--------------------------------------------------------------------*/

static Bool is_non_resumable_signal(Int sigNum)
{
   return (VKI_SIGILL  == sigNum || VKI_SIGBUS  == sigNum ||
           VKI_SIGSEGV == sigNum || VKI_SIGFPE  == sigNum);
}

static void cr_pre_deliver_signal(ThreadId tid, Int sigNum, Bool alt_stack)
{
   // non-resumable signal, must longjmp which we can't necessarily detect,
   // so ignore
   if (is_non_resumable_signal(sigNum))
      return;

   if (0 != in_signal_handler)
      VG_(skin_panic)("can't handle signal interrupting another!\n");

   //VG_(printf)("pre-deliver:  %d\n", sigNum);
   in_signal_handler = -sigNum;
   old_errno = errno;
}

static void cr_post_deliver_signal(ThreadId tid, Int sigNum)
{
   if (is_non_resumable_signal(sigNum))
      return;

   sk_assert(sigNum = in_signal_handler);

   //VG_(printf)("post-deliver: %d\n\n\n", sigNum);
   in_signal_handler = 0;
   if (errno != old_errno)
      errno_changed_error(tid, sigNum, old_errno, errno);
}

//static void handle_longjmp(void)
//{
//   if (0 != in_signal_handler) {
//      VG_(printf)("post-deliver (longjmp): %d\n\n\n", in_signal_handler);
//      in_signal_handler = 0;
//   }
//}

static __attribute__((regparm(0)))
void check_not_in_signal_handler(Char* fnname)
{
   /* Call to the signal handler itself, ignore */
   if      (in_signal_handler < 0)
      in_signal_handler = -in_signal_handler;

   /* In signal handler, oh dear */
   else if (in_signal_handler > 0) {
      if (clo_trace_handlers)
         VG_(message)(Vg_UserMsg, "sig %d: call %s", in_signal_handler, fnname);
      non_reentrant_call_error(fnname, in_signal_handler);
   }
}

static __attribute__((regparm(0)))
void ok_to_be_in_signal_handler(Char* fnname)
{
   /* Call to the signal handler itself, ignore */
   if      (in_signal_handler < 0)
      in_signal_handler = -in_signal_handler;

   else if (in_signal_handler > 0 && clo_trace_handlers)
      VG_(message)(Vg_UserMsg, "sig %d: call %s", in_signal_handler, fnname);
}
   
/*--------------------------------------------------------------------*/
/*--- Instrumentation                                              ---*/
/*--------------------------------------------------------------------*/

// Called at compile-time

/*
Nb: from man page for pthread_mutex_init() (LinuxThreads implementation):
  
  ASYNC-SIGNAL SAFETY
     The mutex functions are not async-signal safe.  What  this
     means is that they should not be called from a signal han
     dler.  In  particular,   calling   pthread_mutex_lock   or
     pthread_mutex_unlock  from  a  signal handler may deadlock
     the calling thread.

Also, from www.burningvoid.com/iaq/posix-signals.html:

     One important safety tip is that pthreads constructs like mutexes,
     condition variables, etc. do not have defined behavior inside signal
     handlers. So you cannot use pthreads calls or synchronize with other
     threads while inside a signal handler.

XXX: maybe condvar functions and other pthread functions should be in the
list too?

*/

static Bool is_definitely_not_reentrant(Char* fnname)
{
   UInt  i;
   Char* bad_funcs[] = {
      // These ones thanks to Steve Grubb
      "_IO_printf", "_IO_puts",
      "syslog", "strdup",
      "__libc_malloc", "__libc_calloc", "__libc_realloc", "__libc_free",
      "inet_ntoa", "gethostbyname", "gethostbyaddr", "getservbynam",
      "readdir", "__readdir", "__readdir64", "strtok",
      "ttyname", "getttyname", "getlogin", "cuserid",
      "getpwent", "getpwnam", "getpwuid",
      "getgrent", "getgrnam", "getgrgid",
      "asctime", "ctime", "gmtime", "localtime",
      "fclose", "fflush", "fflush_unlocked", "fopen", "freopen",
      "fdopen", "fopencookie", "fmemopen", "open_memstream",
      "fprintf", "vfprintf", "fscanf", "vfscanf", "fgetc", "getc", 
      "getc_unlocked", "fgetc_unlocked", "fputc", "putc", "fputc_unlocked", 
      "putc_unlocked", "getw", "putw", "fgets", "fputs", "ungetc", "fseek", 
      "rewind",

      // pthread ones
      "__pthread_mutex_lock", "__pthread_mutex_unlock", 
      "__pthread_mutex_trylock",

      // Found by Tom Hughes <thh@cyberscience.com>, using:
      //   "find /usr/include -name *.h -print0 | xargs -0 grep '_r *('"
      "gethostent",
      "getnetent", "getnetbyname", "getnetbyaddr",
      "getservent", "getservbyport",
      "getprotoent", "getprotobyname", "getprotobynumber",
      "getrpcent", "getrpcbyname", "getrpcbynumber",
      "getmntent",
      "fgetpwent", "fgetgrent",
      "tmpnam",
      "random", "srandom", "initstate", "setstate",
      "drand48", "erand48", "lrand48", "nrand48", "mrand48", "jrand48",
      "srand48", "seed48", "lcong48",
      "ecvt", "fcvt", "qecvt", "qfcvt",
      "strerror",
      "crypt",
      "setkey", "encrypt",
      "hcreate", "hsearch", "hdestroy",

      // Found by me, using the "find" trick above, confirmed by Tom Hughes:
      "ether_ntoa", "ether_aton",
      "gethostbyname2", "getservbyname",
      "rand", "srand",
      "getdate", "getutent", "getutid", "getutline", "pututline",

      // Found by me using the "find" trick, not confirmed, but put them in
      // to err on the side of caution (until proven reentrant...)
      "nis_leaf_of", "nis_name_of", "nis_domain_of", "nis_sperror", 
      "getaliasent", "getaliasbyname", 
      "getnetgrent", 
      "getspent", "getspnam", "sgetspent", "fgetspent", 
      "ptsname ", 
      "BIO_gethostbyname", 

      // end of list
      NULL
   };

   for (i = 0; NULL != bad_funcs[i]; i++)
      if (VG_STREQ(bad_funcs[i], fnname))
         return True;
   
   return False;
}

// XXX: are dynamic linker functions reentrant?  (Surely?)  Are pthread
// functions reentrant?  Is it safe to do IO on a file if it's been locked?
static Bool is_definitely_reentrant(Char* fnname)
{
   UInt i;

   /* From "Advanced Programming in the UNIX Environment", Stevens, p278.
      Those marked with a "+" are not specified by POSIX.1 as reentrant,
      but are listed in the SVR4 SVID. */
   Char* good_funcs[] = { 
      "_exit", "abort"/*+*/, "access", "alarm", 
      "cfgetispeed", "cfgetospeed", "cfsetispeed", "cfsetospeed",
      "chdir", "chmod", "chown", "close", "creat", "dup", "dup2", "execle",
      "execve", "exit"/*+*/, "fcntl", "fork", "fstat", 
      "getegid", "geteuid", "getgid", "getgroups", "getpgrp", "getpid",
      "getppid", "getuid",
      "kill", "link", "longjmp"/*+*/, "lseek", "mkdir", "mkfifo", "open",
      "pathconf", "pause", "pipe", "read", "rename", "rmdir",
      "setgid", "setpgid", "setsid", "setuid",

      "__sigaction", "__libc_sigaction",

      "sigaddset", "sigdelset", "sigemptyset", "sigfillset",
      "sigismember", "signal"/*+*/, "sigpending", 

      "__sigprocmask", 

      "sigsuspend",
      "sleep", "stat", "sysconf",
      "tcdrain", "tcflow", "tcflush", "tcgetattr", "tcgetpgrp", "tcsendbreak",
      "tcsetattr", "tcsetpgrp", "time", "times", "umask", "uname", "unlink",
      "utime", "wait", "waitpid", "write",
      NULL 
   };

   for (i = 0; NULL != good_funcs[i]; i++)
      if (VG_STREQ(good_funcs[i], fnname))
         return True;
   
   return False;
}

static Bool must_check_function(Char* fnname)
{
   if (clo_strict)
      return (! is_definitely_reentrant(fnname));
   else
      return (  is_definitely_not_reentrant(fnname));
}

UCodeBlock* SK_(instrument)(UCodeBlock* cb_in, Addr a)
{
#  define FNNAME_LEN    256
   
   UCodeBlock* cb = VG_(setup_UCodeBlock)(cb_in);
   UInstr*     u;
   UInt        i;
   Char        buf[FNNAME_LEN];
   Char*       fnname;
   Bool        is_entry;

   is_entry = VG_(get_fnname_if_entry)(a, buf, FNNAME_LEN);

   if (is_entry) {
      // copy name... never deallocated
      fnname = VG_(malloc)( (VG_(strlen)(buf)+1) * sizeof(Char) );
      VG_(strcpy)(fnname, buf);

      if ( must_check_function(buf)) {
         // add instrumentation: check this non-reentrant function is not being
         // called from within a signal handler.
         VG_(call_helper_1_0)(cb, (Addr) & check_not_in_signal_handler,
                              (UInt)fnname, 0);
      } else {
         VG_(call_helper_1_0)(cb, (Addr) & ok_to_be_in_signal_handler,
                              (UInt)fnname, 0);
      }
   }

//   if (is_entry && VG_STREQ("__longjmp", buf)) {
//      VG_(call_helper_0_0)(cb, (Addr) & handle_longjmp);
//   }
   
   for (i = 0; i < VG_(get_num_instrs)(cb_in); i++) {
      u = VG_(get_instr)(cb_in, i);
      VG_(copy_UInstr)(cb, u);
   }

   VG_(free_UCodeBlock)(cb_in);

   return cb;
}

/*--------------------------------------------------------------------*/
/*--- Initialisation + finalisation                                ---*/
/*--------------------------------------------------------------------*/

void SK_(pre_clo_init)()
{
   VG_(details_name)            ("Crocus");
   VG_(details_version)         ("0.0.1");
   VG_(details_description)     ("a signal handler checker");
   VG_(details_copyright_author)("Copyright (C) 2003, Nicholas Nethercote.");
   VG_(details_bug_reports_to)  ("njn25@cam.ac.uk");

   /* needs */
   VG_(needs_command_line_options)();

   /* events */
   VG_(track_pre_deliver_signal)  ( cr_pre_deliver_signal  );
   VG_(track_post_deliver_signal) ( cr_post_deliver_signal );

   VG_(register_compact_helper)( (Addr) & check_not_in_signal_handler );
   VG_(register_compact_helper)( (Addr) & ok_to_be_in_signal_handler );
//   VG_(register_compact_helper)( (Addr) & handle_longjmp );
}

void SK_(post_clo_init)(void)
{
}

void SK_(fini)(exitcode)
{
}

/*--------------------------------------------------------------------*/
/*--- end                                                ex_main.c ---*/
/*--------------------------------------------------------------------*/
