
/*--------------------------------------------------------------------*/
/*--- Annelid: a pointer-use checker.                    an_main.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Annelid, a Valgrind skin for checking pointer
   use in programs.

   Copyright (C) 2003 Nicholas Nethercote
      njn25@cam.ac.uk

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

// XXX: recycle freed segments

//--------------------------------------------------------------
// Metadata:
//   HeapBlock.id :: Seg (stored as heap shadowchunk; always non-zero)
//   MemLoc.aseg  :: Seg (implicitly stored)
//   MemLoc.vseg  :: Seg (explicitly stored as the shadow memory)
//   RegLoc.vseg  :: Seg (explicitly stored as shadow registers)
//
// A Seg is made when new memory is created, eg. with malloc() or mmap().
// There are two other Segs:
//  - NONPTR:  for something that's definitely not a pointer
//  - UNKNOWN: for something that could be a pointer
//  - BOTTOM:  used with pointer differences (see below)
//
// MemLoc.vseg is done at word granularity.  If a pointer is written
// to memory misaligned, the information about it will be lost -- it's
// treated as two sub-word writes to two adjacent words.  This avoids
// certain nasty cases that could arise if we tried to track unaligned
// pointers.  Fortunately, misalignment is rare so we don't lose much
// information this way.
//
// MemLoc.aseg is done at byte granularity, and *implicitly* -- ie. not
// directly accessible like MemLoc.vseg, but only by searching through all
// the segments.  Fortunately, it's mostly checked at LOADs/STOREs;  at that
// point we have a pointer p to the MemLoc m as the other arg of the
// LOAD/STORE, so we can check to see if the p.vseg's range includes m.  If
// not, it's an error and we have to search through all segments to find out
// what m.aseg really is.  That's still pretty fast though, thanks to the
// interval skip-list used.  With syscalls we must also do the skip-list
// search, but only on the first and last bytes touched.
//--------------------------------------------------------------

//--------------------------------------------------------------
// Assumptions, etc:
// - see comment at top of SK_(instrument)() for how sub-word ops are
//   handled.
//
// - ioctl(), socketcall() (and ipc() will be) assumed to return non-pointers
//
// - FPU_W is assumed to never write pointers.
//
// - Assuming none of the post_mem_writes create segments worth tracking.
//
// - Treating mmap'd segments (all! including code) like heap segments.  But
//   their ranges can change, new ones can be created by unmapping parts of
//   old segments, etc.  But this nasty behaviour seems to never happen -- 
//   there are assertions checking it.
//--------------------------------------------------------------

//--------------------------------------------------------------
// What I am checking:
// - Type errors:
//    * ADD, OR, LEA2: error if two pointer inputs.
//    * ADC, SBB: error if one or two pointer inputs.
//    * AND, OR: error if two unequal pointer inputs.
//    * NEG: error if pointer input.
//    * {,i}mul_32_64 if either input is a pointer.
//    * shldl/shrdl, bsf/bsr if any inputs are pointers.
//
// - LOAD, STORE:
//    * ptr.vseg must match ptee.aseg.
//    * ptee.aseg must not be a freed segment.
//
// - syscalls: for those accessing memory, look at first and last bytes:
//    * check first.aseg == last.aseg
//    * check first.aseg and last.aseg are not freed segments.
//
// What I am not checking, that I expected to when I started:
// - AND, XOR: allowing two pointers to be used if both from the same segment,
//   because "xor %r,%r" is commonly used to zero %r, and "test %r,%r"
//   (which is translated with an AND) is common too.
//
// - div_64_32/idiv_64_32 can take pointer inputs for the dividend;
//   division doesn't make sense, but modulo does, and they're done with the
//   same instruction.  (Could try to be super-clever and watch the outputs
//   to see if the quotient is used, but not worth it.)
//
// - mul_64_32/imul_64_32 can take pointers inputs for one arg or the
//   other, but not both.  This is because some programs (eg. Mozilla
//   Firebird) multiply pointers in hash routines.
//
// - NEG: can take a pointer.  It happens in glibc in a few places.  I've
//   seen the code, didn't understand it, but it's done deliberately.
//
// What I am not checking/doing, but could, but it would require more
// instrumentation and/or slow things down a bit:
// - SUB: when differencing two pointers, result is BOTTOM, ie. "don't
//   check".  Could link segments instead, slower but a bit more accurate.
//   Also use BOTTOM when doing (ptr - unknown), which could be a pointer
//   difference with a stack/static pointer.
//
// - PUTF: input should be non-pointer
//
// - arithmetic error messages: eg. for adding two pointers, just giving the
//   segments, not the actual pointers.
//
// What I am not checking, and would be difficult:
// - mmap(...MAP_FIXED...) is not handled specially.  It might be used in
//   ways that fool Annelid into giving false positives.
//
// - syscalls: for those accessing memory, not checking that the asegs of the
//   accessed words match the vseg of the accessing pointer, because the
//   vseg is not easily accessible at the required time (would required
//   knowing for every syscall which register each arg came in, and looking
//   there).
//
// What I am not checking, and would be difficult, but doesn't matter:
// - free(p): similar to syscalls, not checking that the p.vseg matches the
//   aseg of the first byte in the block.  However, Memcheck does an
//   equivalent "bad free" check using shadow_chunks;  indeed, Annelid could
//   do the same check, but there's no point duplicating functionality.  So
//   no loss, really.
//
// Other:
// - not doing anything with mprotect();  probably not worth the effort.
//--------------------------------------------------------------

//--------------------------------------------------------------
// Todo:
// - Segments for stack frames.  Would detect (some, large) stack
//   over/under-runs, dangling pointers.
//
// - Segments for static data.  Would detect over/under-runs.  Requires
//   reading debug info.
//--------------------------------------------------------------

//--------------------------------------------------------------
// Some profiling results:
//                                                 twolf   konq    date sz
// 1. started                                              35.0s   14.7
// 2. introduced GETV/PUTV                                 30.2s   10.1
// 3. inlined check_load_or_store                  5.6s    27.5s   10.1
// 4. (made check_load, check_store4 regparm(0))          (27.9s) (11.0)
// 5. um, not sure                                 5.3s    27.3s   10.6
//    ...
// 6. after big changes, corrections              11.2s    32.8s   14.0
// 7. removed link-segment chasing in check/L/S    8.9s    30.8s   14.0
// 8. avoiding do_lea1 if k is a nonptr            8.0s    28.0s   12.9
//--------------------------------------------------------------

//#include "vg_skin.h"

#include "an_list.h"

//#include "vg_profile.c"

VG_DETERMINE_INTERFACE_VERSION

/*------------------------------------------------------------*/
/*--- Profiling events                                     ---*/
/*------------------------------------------------------------*/

typedef
   enum {
      VgpGetMemAseg = VgpFini+1,
      VgpCheckLoadStore,
      VgpCheckLoad4,
      VgpCheckLoad21,
      VgpCheckStore4,
      VgpCheckStore21,
      VgpCheckFpuR,
      VgpCheckFpuW,
      VgpDoAdd,
      VgpDoSub,
      VgpDoAdcSbb,
      VgpDoXor,
      VgpDoAnd,
      VgpDoOr,
      VgpDoLea1,
      VgpDoLea2,
   }
   VgpSkinCC;

/*------------------------------------------------------------*/
/*--- Command line options                                 ---*/
/*------------------------------------------------------------*/

Bool clo_partial_loads_ok = True;

Bool SK_(process_cmd_line_option)(Char* arg)
{
   if      (VG_CLO_STREQ(arg, "--partial-loads-ok=yes"))
      clo_partial_loads_ok = True;
   else if (VG_CLO_STREQ(arg, "--partial-loads-ok=no"))
      clo_partial_loads_ok = False;

   else
      return VG_(replacement_malloc_process_cmd_line_option)(arg);

   return True;
}

void SK_(print_usage)(void)
{
   VG_(printf)(
"    --partial-loads-ok=no|yes same as for Memcheck [yes]\n"
   );
   VG_(replacement_malloc_print_usage)();
}

void SK_(print_debug_usage)(void)
{
   VG_(replacement_malloc_print_debug_usage)();
}

/*------------------------------------------------------------*/
/*--- Segments                                             ---*/
/*------------------------------------------------------------*/

// Choose values that couldn't possibly be pointers
#define NONPTR          ((Seg)0xA1)
#define UNKNOWN         ((Seg)0xB2)
#define BOTTOM          ((Seg)0xC3)

static ISList* seglist = NULL;

// So that post_reg_write_clientcall knows the segment just allocated.
static Seg last_seg_added;

// Returns the added heap segment
static Seg add_new_segment ( Addr p, UInt size, SegStatus status )
{
   ExeContext* where = VG_(get_ExeContext)( VG_(get_current_or_recent_tid)() );
   Seg         seg   = Seg__construct(p, size, where, status);

   last_seg_added = seg;

   ISList__insertI( seglist, seg );

   return seg;
}

// Forward declarations
static void copy_mem( Addr from, Addr to, UInt len );
static void set_mem_unknown ( Addr a, UInt len );

static __inline__
void* alloc_and_new_mem_heap ( Int size, UInt alignment, Bool is_zeroed )
{
   Addr p;

   if (size < 0) return NULL;

   p = (Addr)VG_(cli_malloc)(alignment, size);
   if (is_zeroed) VG_(memset)((void*)p, 0, size);

   set_mem_unknown( p, size );
   add_new_segment( p, size, SegHeap );

   return (void*)p;
}

static void die_and_free_mem_heap ( Seg seg )
{
   // Empty and free the actual block, if on the heap (not necessary for
   // mmap segments).
   set_mem_unknown( Seg__a(seg), Seg__size(seg) );
   VG_(cli_free)( (void*)Seg__a(seg) );

   if (0 == Seg__size(seg)) {
      // XXX: can recycle Seg now
   }

   // Remember where freed
   Seg__heap_free( seg, 
                   VG_(get_ExeContext)( VG_(get_current_or_recent_tid)() ) );
}

#if 0
static void die_and_free_mem_munmap ( Seg seg/*, Seg* prev_chunks_next_ptr*/ )
{
   // Remember where freed
   seg->where = VG_(get_ExeContext)( VG_(get_current_or_recent_tid)() );
   sk_assert(SegMmap == seg->status);
   seg->status = SegMmapFree;
}
#endif

static __inline__ void handle_free_heap( void* p )
{
   Seg seg;
   if ( ! ISList__findI0( seglist, (Addr)p, &seg ) )
      return;

   die_and_free_mem_heap( seg );
}

#if 0
static void handle_free_munmap( void* p, UInt len )
{
   Seg seg;
   if ( ! ISList__findI( seglist, (Addr)p, &seg ) ) {
      VG_(skin_panic)("handle_free_munmap:  didn't find segment;\n"
                      "should check all the mmap segment ranges, this\n"
                      "one should be in them.  (Well, it's possible that\n"
                      "it's not, but I'm not handling that case yet.)\n");
   }
   if (len != VG_ROUNDUP(Seg__size(seg), VKI_BYTES_PER_PAGE)) {
//      if (seg->is_truncated_map && seg->size < len) {
//         // ok
//      } else {
         VG_(printf)("len = %u, seg->size = %u\n", len, Seg__size(seg));
         VG_(skin_panic)("handle_free_munmap:  length didn't match\n");
//      }
   }

   die_and_free_mem_munmap( seg/*, prev_chunks_next_ptr*/ );
}
#endif
/*------------------------------------------------------------*/
/*--- Shadow memory                                        ---*/
/*------------------------------------------------------------*/

__attribute__((unused))
static void pp_curr_ExeContext(void)
{
   VG_(pp_ExeContext)(
      VG_(get_ExeContext)(
         VG_(get_current_or_recent_tid)() ) );
   VG_(message)(Vg_UserMsg, "");
}

#define SEC_MAP_WORDS  16384   /* Words per secondary map */

typedef
   struct {
      Seg vseg[SEC_MAP_WORDS];
   }
   SecMap;

static SecMap* primary_map[ 65536 ];
static SecMap  distinguished_secondary_map;

static void init_shadow_memory ( void )
{
   Int i;

   for (i = 0; i < SEC_MAP_WORDS; i++)
      distinguished_secondary_map.vseg[i] = UNKNOWN;

   /* These entries gradually overwritten as used address space expands. */
   for (i = 0; i < 65536; i++)
      primary_map[i] = &distinguished_secondary_map;
}

static SecMap* alloc_secondary_map ( Char* caller )
{
   SecMap* map;
   UInt  i;

   sk_assert(0 == (sizeof(SecMap) % VKI_BYTES_PER_PAGE));
   map = VG_(get_memory_from_mmap)( sizeof(SecMap), caller );

   for (i = 0; i < SEC_MAP_WORDS; i++)
      map->vseg[i] = UNKNOWN;

   return map;
}

#define IS_DISTINGUISHED_SM(smap) \
   ((smap) == &distinguished_secondary_map)

#define ENSURE_MAPPABLE(addr,caller)                                  \
   do {                                                               \
      if (IS_DISTINGUISHED_SM(primary_map[(addr) >> 16])) {       \
         primary_map[(addr) >> 16] = alloc_secondary_map(caller);     \
         /*VG_(printf)("new 2map because of %p\n", addr);*/           \
      } \
   } while(0)

// Nb: 'a' must be aligned.
static __inline__
Seg get_mem_vseg ( Addr a )
{
   /* Use bits 31..16 for primary, 15..2 for secondary lookup */
   SecMap* sm     = primary_map[a >> 16];
   UInt    sm_off = (a & 0xFFFC) >> 2;
   sk_assert(IS_ALIGNED4(a));
#if 0
   if (IS_DISTINGUISHED_SM(sm)) {
      VG_(printf)("accessed distinguished 2ndary map! 0x%x\n", a);
   }
#endif
   return sm->vseg[sm_off];
}

// Nb 'a' must be aligned.
static __inline__
void set_mem_vseg ( Addr a, Seg vseg )
{
   SecMap* sm;

   sk_assert(IS_ALIGNED4(a));
   ENSURE_MAPPABLE(a, "set_mem_vseg");

   /* Use bits 31..16 for primary, 15..2 for secondary lookup */
   sm = primary_map[a >> 16];
   if (IS_DISTINGUISHED_SM(sm)) {
      VG_(printf)("wrote to address 0x%x\n", a);
      VG_(printf)("warning: ignoring write to dist. 2ndary map 0x%x\n", a);
      return;
   }
   sm->vseg[(a & 0xFFFC) >> 2] = vseg;
}

// Returns UNKNOWN if no matches.  Never returns BOTTOM or NONPTR.
static Seg get_mem_aseg( Addr a )
{
   Seg aseg;
   Bool is_found;

   VGP_PUSHCC(VgpGetMemAseg);
   is_found = ISList__findI( seglist, a, &aseg );
   VGP_POPCC(VgpGetMemAseg);
   return ( is_found ? aseg : UNKNOWN );
}

/*--------------------------------------------------------------------*/
/*--- Error handling                                               ---*/
/*--------------------------------------------------------------------*/

typedef
   enum {
      /* Possible data race */
      LoadStoreSupp,
      ArithSupp,
      SysParamSupp,
   }
   AnnelidSuppKind;

/* What kind of error it is. */
typedef
   enum {
      LoadStoreErr,     // mismatched ptr/addr segments on load/store
      ArithErr,         // bad arithmetic between two segment pointers
      SysParamErr,      // block straddling >1 segment passed to syscall
   }
   AnnelidErrorKind;


// These ones called from generated code.

typedef
   struct {
      Addr a;
      UInt size;
      Seg  vseg;
      Bool is_write;
   }
   LoadStoreExtra;

typedef
   struct {
      UInt   arg1;
      UInt   arg2;
      Seg    seg1;
      Seg    seg2;
      Int    opcode; // If +ve, a UOpcode;  if -ve, an asm offset.  Yuk.
   }
   ArithExtra;

typedef
   struct {
      CorePart part;
      Addr lo;
      Addr hi;
      Seg  seglo;
      Seg  seghi;
   }
   SysParamExtra;

static UInt actual_arg1, actual_arg2/*, actual_res*/;

static
void record_loadstore_error( Addr a, UInt size, Seg vseg, Bool is_write )
{
   LoadStoreExtra extra = {
      .a = a, .size = size, .vseg = vseg, .is_write = is_write
   };
   VG_(maybe_record_error)( VG_(get_current_or_recent_tid)(), LoadStoreErr,
                            /*a*/0, /*str*/NULL, /*extra*/(void*)&extra);
}

static void
record_arith_error( UInt arg1, UInt arg2, Seg seg1, Seg seg2, UInt opcode )
{
   ArithExtra extra = {
      .arg1 = arg1, .arg2 = arg2, .seg1 = seg1, .seg2 = seg2, .opcode = opcode
   };
   VG_(maybe_record_error)( VG_(get_current_or_recent_tid)(), ArithErr,
                            /*a*/0, /*str*/NULL, /*extra*/(void*)&extra);
}

// This one called from non-generated code.

static void record_sysparam_error( ThreadId tid, CorePart part, Char* s,
                                   Addr lo, Addr hi, Seg seglo, Seg seghi )
{
   SysParamExtra extra = {
      .part = part, .lo = lo, .hi = hi, .seglo = seglo, .seghi = seghi
   };
   VG_(maybe_record_error)( tid, SysParamErr, /*a*/(Addr)0, /*str*/s,
                            /*extra*/(void*)&extra);
}


Bool SK_(eq_SkinError) ( VgRes res, Error* e1, Error* e2 )
{
   sk_assert(VG_(get_error_kind)(e1) == VG_(get_error_kind)(e2));

   // Nb: ok to compare string pointers, rather than string contents,
   // because the same static strings are shared.

   switch (VG_(get_error_kind)(e1)) {

   case LoadStoreErr:
   case SysParamErr:
      return VG_(get_error_string)(e1) == VG_(get_error_string)(e2);

   case ArithErr:
      return True;

   default:
      VG_(skin_panic)("eq_SkinError: unrecognised error kind");
   }
}

static Char* readwrite(Bool is_write)
{
   return ( is_write ? "write" : "read" );
}

static __inline__
Bool is_known_segment(Seg seg)
{
   return (UNKNOWN != seg && BOTTOM != seg && NONPTR != seg);
}

static Char* an_name_UOpcode(Int opc)
{
   if (opc >= 0) return VG_(name_UOpcode)(True, opc);

   else if (-opc == VGOFF_(helper_mul_32_64))    return  "MUL";
   else if (-opc == VGOFF_(helper_imul_32_64))   return "IMUL";
   else if (-opc == VGOFF_(helper_div_64_32))    return  "DIV";
   else if (-opc == VGOFF_(helper_idiv_64_32))   return "IDIV";

   else VG_(skin_panic)("an_name_UOpcode");
}
      
void SK_(pp_SkinError) ( Error* err )
{
   switch (VG_(get_error_kind)(err)) {
   //----------------------------------------------------------
   case LoadStoreErr: {
      LoadStoreExtra* extra = (LoadStoreExtra*)VG_(get_error_extra)(err);
      Char           *place, *legit, *how_invalid;
      Addr            a    = extra->a;
      Seg             vseg = extra->vseg;

      sk_assert(is_known_segment(vseg) || NONPTR == vseg);

      if (NONPTR == vseg) {
         // Access via a non-pointer
         VG_(message)(Vg_UserMsg, "Invalid %s of size %u",
                                   readwrite(extra->is_write), extra->size);
         VG_(pp_ExeContext)( VG_(get_error_where)(err) );
         VG_(message)(Vg_UserMsg,
                      "Address %p is not derived from any known block", a);

      } else {
         // Access via a pointer, but outside its range.
         Int cmp, miss_size;
         Seg__cmp(vseg, a, &cmp, &miss_size);
         if      (cmp  < 0) place = "before";
         else if (cmp == 0) place = "inside";
         else               place = "after";
         how_invalid = ( ( Seg__is_freed(vseg) && 0 != cmp )
                       ? "Doubly-invalid" : "Invalid" );
         legit = ( Seg__is_freed(vseg) ? "once-" : "" );

         VG_(message)(Vg_UserMsg, "%s %s of size %u", how_invalid,
                                  readwrite(extra->is_write), extra->size);
         VG_(pp_ExeContext)( VG_(get_error_where)(err) );

         VG_(message)(Vg_UserMsg,
                      "Address %p is %d bytes %s the accessing pointer's",
                      a, miss_size, place);
         VG_(message)(Vg_UserMsg,
                    " %slegitimate range, the %d-byte block %s",
                      legit, Seg__size(vseg), Seg__status_str(vseg) );
         VG_(pp_ExeContext)(Seg__where(vseg));
      }
      break;
   }

   //----------------------------------------------------------
   case ArithErr: {
      ArithExtra* extra = VG_(get_error_extra)(err);
//      UInt   arg1   = extra->arg1;
//      UInt   arg2   = extra->arg2;
      Seg    seg1   = extra->seg1;
      Seg    seg2   = extra->seg2;
      Char*  which;

      sk_assert(BOTTOM != seg1);
      sk_assert(BOTTOM != seg2 && UNKNOWN != seg2);

//      VG_(message)(Vg_UserMsg, "First %p, second %p", arg1, arg2);

      VG_(message)(Vg_UserMsg, "Invalid %s", an_name_UOpcode(extra->opcode));
      VG_(pp_ExeContext)( VG_(get_error_where)(err) );

      if (seg1 != seg2) {
         if (NONPTR == seg1) {
            VG_(message)(Vg_UserMsg, "First arg not a pointer");
         } else if (UNKNOWN == seg1) {
            VG_(message)(Vg_UserMsg, "First arg may be a pointer");
         } else {
            VG_(message)(Vg_UserMsg, "First arg derived from address %p of "
                                     "%d-byte block %s",
                                     Seg__a(seg1), Seg__size(seg1),
                                     Seg__status_str(seg1) );
            VG_(pp_ExeContext)(Seg__where(seg1));
         }
         which = "Second arg";
      } else {
         which = "Both args";
      }
      if (NONPTR == seg2) {
         VG_(message)(Vg_UserMsg, "%s not a pointer", which);
      } else {
         VG_(message)(Vg_UserMsg, "%s derived from address %p of "
                                  "%d-byte block %s",
                                  which, Seg__a(seg2), Seg__size(seg2),
                                  Seg__status_str(seg2));
         VG_(pp_ExeContext)(Seg__where(seg2));
      }
      break;
   }

   //----------------------------------------------------------
   case SysParamErr: {
      SysParamExtra* extra = (SysParamExtra*)VG_(get_error_extra)(err);
      Addr           lo    = extra->lo;
      Addr           hi    = extra->hi;
      Seg            seglo = extra->seglo;
      Seg            seghi = extra->seghi;
      Char*          s     = VG_(get_error_string) (err);
      Char*          what;

      sk_assert(BOTTOM != seglo && BOTTOM != seghi);

      if      (Vg_CoreSysCall == extra->part) what = "Syscall param ";
      else if (Vg_CorePThread == extra->part) what = "";
      else    VG_(skin_panic)("bad CorePart");

      if (seglo == seghi) {
         // freed block
         sk_assert(is_known_segment(seglo) && Seg__is_freed(seglo));
         VG_(message)(Vg_UserMsg, "%s%s contains unaddressable byte(s)", what, s);
         VG_(pp_ExeContext)( VG_(get_error_where)(err) );

         VG_(message)(Vg_UserMsg, "Address %p is %d bytes within a "
                                  "%d-byte block %s",
                                  lo, lo-Seg__a(seglo), Seg__size(seglo),
                                  Seg__status_str(seglo) );
         VG_(pp_ExeContext)(Seg__where(seglo));

      } else {
         // mismatch
         VG_(message)(Vg_UserMsg, "%s%s is non-contiguous", what, s);
         VG_(pp_ExeContext)( VG_(get_error_where)(err) );

         if (UNKNOWN == seglo) {
            VG_(message)(Vg_UserMsg, "First byte is not within a known block");
         } else {
            VG_(message)(Vg_UserMsg, "First byte (%p) is %d bytes within a "
                                     "%d-byte block %s",
                                     lo, lo-Seg__a(seglo), Seg__size(seglo),
                                     Seg__status_str(seglo) );
            VG_(pp_ExeContext)(Seg__where(seglo));
         }

         if (UNKNOWN == seghi) {
            VG_(message)(Vg_UserMsg, "Last byte is not within a known block");
         } else {
            VG_(message)(Vg_UserMsg, "Last byte (%p) is %d bytes within a "
                                     "%d-byte block %s",
                                     hi, hi-Seg__a(seghi), Seg__size(seghi),
                                     Seg__status_str(seghi));
            VG_(pp_ExeContext)(Seg__where(seghi));
         }
      }
      break;
   }

   default:
      VG_(skin_panic)("pp_SkinError: unrecognised error kind");
   }
}

UInt SK_(update_extra) ( Error* err )
{
   switch (VG_(get_error_kind)(err)) {
   case LoadStoreErr: return sizeof(LoadStoreExtra);
   case ArithErr:     return 0;
   case SysParamErr:  return sizeof(SysParamExtra);
   default:           VG_(skin_panic)("update_extra");
   }
}

Bool SK_(recognised_suppression) ( Char* name, Supp *su )
{
   SuppKind skind;

   if      (VG_STREQ(name, "LoadStore"))  skind = LoadStoreSupp;
   else if (VG_STREQ(name, "Arith"))      skind = ArithSupp;
   else if (VG_STREQ(name, "SysParam"))   skind = SysParamSupp;
   else
      return False;

   VG_(set_supp_kind)(su, skind);
   return True;
}

Bool SK_(read_extra_suppression_info) ( Int fd, Char* buf, Int nBuf, Supp* su )
{
   Bool eof;

   if (VG_(get_supp_kind)(su) == SysParamSupp) {
      eof = VG_(get_line) ( fd, buf, nBuf );
      if (eof) return False;
      VG_(set_supp_string)(su, VG_(strdup)(buf));
   }
   return True;
}

Bool SK_(error_matches_suppression)(Error* err, Supp* su)
{
   ErrorKind  ekind     = VG_(get_error_kind )(err);

   switch (VG_(get_supp_kind)(su)) {
   case LoadStoreSupp:  return (ekind == LoadStoreErr);
   case ArithSupp:      return (ekind == ArithErr);
   case SysParamSupp:   return (ekind == SysParamErr);
   default:
      VG_(printf)("Error:\n"
                  "  unknown suppression type %d\n",
                  VG_(get_supp_kind)(su));
      VG_(skin_panic)("unknown suppression type in "
                      "SK_(error_matches_suppression)");
   }
}

Char* SK_(get_error_name) ( Error* err )
{
   switch (VG_(get_error_kind)(err)) {
   case LoadStoreErr:       return "LoadStore";
   case ArithErr:           return "Arith";
   case SysParamErr:        return "SysParam";
   default:                 VG_(skin_panic)("get_error_name: unexpected type");
   }
}

void SK_(print_extra_suppression_info) ( Error* err )
{
   if (SysParamErr == VG_(get_error_kind)(err)) {
      VG_(printf)("   %s\n", VG_(get_error_string)(err));
   }
}

/*------------------------------------------------------------*/
/*--- malloc() et al replacements                          ---*/
/*------------------------------------------------------------*/

void* SK_(malloc) ( Int n )
{
   return alloc_and_new_mem_heap ( n, VG_(clo_alignment), /*is_zeroed*/False );
}

void* SK_(__builtin_new) ( Int n )
{
   return alloc_and_new_mem_heap ( n, VG_(clo_alignment), /*is_zeroed*/False );
}

void* SK_(__builtin_vec_new) ( Int n )
{
   return alloc_and_new_mem_heap ( n, VG_(clo_alignment), /*is_zeroed*/False );
}

void* SK_(calloc) ( Int nmemb, Int size )
{
   return alloc_and_new_mem_heap ( nmemb*size, VG_(clo_alignment),
                              /*is_zeroed*/True );
}

void* SK_(memalign) ( Int align, Int n )
{
   return alloc_and_new_mem_heap ( n, align,              /*is_zeroed*/False );
}

void SK_(free) ( void* p )
{
   // Should arguably check here if p.vseg matches the segID of the
   // pointed-to block... unfortunately, by this stage, we don't know what
   // p.vseg is, because we don't know the address of p (the p here is a
   // copy, and we've lost the address of its source).  To do so would
   // require passing &p in, which would require rewriting part of
   // vg_replace_malloc.c... argh.
   //
   // However, Memcheck does free checking, and will catch almost all
   // violations this checking would have caught.  (Would only miss if we
   // unluckily passed an unrelated pointer to the very start of a heap
   // block that was unrelated to that block.  This is very unlikely!)    So
   // we haven't lost much.

   handle_free_heap(p);
}

void SK_(__builtin_delete) ( void* p )
{
   handle_free_heap(p);
}

void SK_(__builtin_vec_delete) ( void* p )
{
   handle_free_heap(p);
}

void* SK_(realloc) ( void* p_old, Int new_size )
{
   Seg seg;

   /* First try and find the block. */
   if ( ! ISList__findI0( seglist, (Addr)p_old, &seg ) )
      return NULL;

   sk_assert(Seg__a(seg) == (Addr)p_old);

   if (new_size <= Seg__size(seg)) {
      /* new size is smaller */
      sk_assert(new_size > 0);
      set_mem_unknown( Seg__a(seg)+new_size, Seg__size(seg)-new_size );
      Seg__resize(seg, new_size, 
                  VG_(get_ExeContext)( VG_(get_current_or_recent_tid)() ) );
      last_seg_added = seg;      // necessary for post_reg_write_clientcall
      return p_old;

   } else {
      /* new size is bigger: allocate, copy from old to new */
      Addr p_new = (Addr)VG_(cli_malloc)(VG_(clo_alignment), new_size);
      VG_(memcpy)((void*)p_new, p_old, Seg__size(seg));

      /* Notification: first half kept and copied, second half new */
      copy_mem       ( (Addr)p_old, p_new, Seg__size(seg) );
      set_mem_unknown( p_new+Seg__size(seg), new_size-Seg__size(seg) );

      /* Free old memory */
      die_and_free_mem_heap( seg );

      /* This has to be after die_and_free_mem_heap, otherwise the
         former succeeds in shorting out the new block, not the
         old, in the case when both are on the same list.  */
      add_new_segment ( p_new, new_size, SegHeap );

      return (void*)p_new;
   }
}

/*------------------------------------------------------------*/
/*--- Memory events                                        ---*/
/*------------------------------------------------------------*/

static __inline__
void set_mem( Addr a, UInt len, Seg seg )
{
   Addr end;

   if (0 == len)
      return;

   if (len > 100 * 1000 * 1000)
      VG_(message)(Vg_UserMsg,
                   "Warning: set address range state: large range %d", len);

   a   = VG_ROUNDDN(a,       4);
   end = VG_ROUNDUP(a + len, 4);
   for ( ; a < end; a += 4)
      set_mem_vseg(a, seg);
}

static void set_mem_unknown( Addr a, UInt len )
{
   set_mem( a, len, UNKNOWN );
}

static void set_mem_nonptr( Addr a, UInt len )
{
   set_mem( a, len, NONPTR );
}

static void new_mem_startup( Addr a, UInt len, Bool rr, Bool ww, Bool xx )
{
   set_mem_unknown( a, len );
}

// XXX: Currently not doing anything with brk() -- new segments, or not?
// Proper way to do it would be to grow/shrink a single, special brk segment.
//
// brk is difficult: it defines a single segment, of changeable size.
// It starts off with size zero, at the address given by brk(0).  There are
// no pointers within the program to it.  Any subsequent calls by the
// program to brk() (possibly growing or shrinking it) return pointers to
// the *end* of the segment (nb: this is the kernel brk(), which is
// different to the libc brk()).
//
// If fixing this, don't forget to update the brk case in SK_(post_syscall).
//
// Nb: not sure if the return value is the last byte addressible, or one
// past the end of the segment.
//
static void new_mem_brk( Addr a, UInt len )
{
   set_mem_unknown(a, len);
   //VG_(skin_panic)("can't handle new_mem_brk");
}

// Not quite right:  if you mmap a segment into a specified place, it could
// be legitimate to do certain arithmetic with the pointer that it wouldn't
// otherwise.  Hopefully this is rare, though.
static void new_mem_mmap( Addr a, UInt len, Bool rr, Bool ww, Bool xx )
{
#if 0
   Seg seg = NULL;

   // Check for overlapping segments
#if 0
   is_overlapping_seg___a   = a;    // 'free' variable
   is_overlapping_seg___len = len;  // 'free' variable
   seg = (Seg)VG_(HT_first_match) ( mlist, is_overlapping_seg );
   is_overlapping_seg___a   = 0;    // paranoia, reset
   is_overlapping_seg___len = 0;    // paranoia, reset
#endif

   // XXX: do this check properly with ISLists

   if ( ISList__findI( seglist, a, &seg )) {
      sk_assert(SegMmap == seg->status || SegMmapFree == seg->status);
      if (SegMmap == seg->status)
   
   }

   if (NULL != seg) {
      // Right, we found an overlap
      if (VG_(clo_verbosity) > 1)
         VG_(message)(Vg_UserMsg, "mmap overlap:  old: %p, %d;  new: %p, %d",
                                  seg->left, Seg__size(seg), a, len);
      if (seg->left <= a && a <= seg->right) {
         // New one truncates end of the old one.  Nb: we don't adjust its
         // size, because the first segment's pointer can be (and for
         // Konqueror, is) legitimately used to access parts of the second
         // segment.  At least, I assume Konqueror is doing something legal.
         // so that a size mismatch upon munmap isn't a problem.
//         seg->size = a - seg->data;
//         seg->is_truncated_map = True;
//         if (VG_(clo_verbosity) > 1)
//            VG_(message)(Vg_UserMsg, "old seg truncated to length %d",
//                                     seg->size);
      } else {
         VG_(skin_panic)("Can't handle this mmap() overlap case");
      }
   }
   set_mem_unknown( a, len );
   add_new_segment( a, len, SegMmap );
#endif
}

static void copy_mem( Addr from, Addr to, UInt len )
{
   Addr fromend = from + len;

   // Must be aligned due to malloc always returning aligned objects.
   sk_assert(IS_ALIGNED4(from) && IS_ALIGNED4(to));

   // Must only be called with positive len.
   if (0 == len)
      return;

   for ( ; from < fromend; from += 4, to += 4)
      set_mem_vseg( to, get_mem_vseg(from) );
}

// Similar to SK_(realloc)()
static void copy_mem_remap( Addr from, Addr to, UInt len )
{
   VG_(skin_panic)("argh: copy_mem_remap");
}

static void die_mem_brk( Addr a, UInt len )
{
   set_mem_unknown(a, len);
//   VG_(skin_panic)("can't handle die_mem_brk()");
}

static void die_mem_munmap( Addr a, UInt len )
{
//   handle_free_munmap( (void*)a, len );
}

// Don't need to check all addresses within the block; in the absence of
// discontiguous segments, the segments for the first and last bytes should
// be the same.  Can't easily check the pointer segment matches the block
// segment, unfortunately, but the first/last check should catch most
// errors.
static void pre_mem_access2 ( CorePart part, ThreadId tid, Char* s,
                              Addr lo, Addr hi )
{
   Seg seglo, seghi;

   // Don't check code being translated -- very slow, and not much point
   if (Vg_CoreTranslate == part) return;

   // Don't check the signal case -- only happens in core, no need to check
   if (Vg_CoreSignal == part) return;

   // Check first and last bytes match
   seglo = get_mem_aseg( lo );
   seghi = get_mem_aseg( hi );
   sk_assert( BOTTOM != seglo && NONPTR != seglo );
   sk_assert( BOTTOM != seghi && NONPTR != seghi );

   if ( seglo != seghi || (UNKNOWN != seglo && Seg__is_freed(seglo)) ) {
      // First/last bytes don't match, or seg has been freed.
      switch (part) {
      case Vg_CoreSysCall:
      case Vg_CorePThread:
         record_sysparam_error(tid, part, s, lo, hi, seglo, seghi);
         break;

      default:
         VG_(printf)("part = %d\n", part);
         VG_(skin_panic)("unknown corepart in pre_mem_access2");
      }
   }
}

static void pre_mem_access ( CorePart part, ThreadId tid, Char* s,
                             Addr base, UInt size )
{
   pre_mem_access2( part, tid, s, base, base + size - 1 );
}

static
void pre_mem_read_asciiz ( CorePart part, ThreadId tid, Char* s, Addr lo )
{
   Addr hi = lo;

   // Nb: the '\0' must be included in the lo...hi range
   while ('\0' != *(Char*)hi) hi++;
   pre_mem_access2( part, tid, s, lo, hi );
}

static void post_mem_write(Addr a, UInt len)
{
   set_mem_unknown(a, len);
}

/*------------------------------------------------------------*/
/*--- Register event handlers                              ---*/
/*------------------------------------------------------------*/

static void post_regs_write_init ( void )
{
   UInt i;
   for (i = R_EAX; i <= R_EDI; i++)
      VG_(set_shadow_archreg)( i, (UInt)UNKNOWN );

   // Don't bother about eflags
}

static void post_reg_write_nonptr(ThreadId tid, UInt reg)
{
   // syscall_return: Default is non-pointer.  If it really is a pointer
   // (eg. for mmap()), SK_(post_syscall) sets it again afterwards.
   //
   // clientreq_return: All the global client requests return non-pointers
   // (except possibly CLIENT_CALL[0123], but they're handled by
   // post_reg_write_clientcall, not here).
   //
   VG_(set_thread_shadow_archreg)( tid, reg, (UInt)NONPTR );
}

static void post_reg_write_nonptr_or_unknown(ThreadId tid, UInt reg)
{
   // deliver_signal: called from two places; one sets the reg to zero, the
   // other sets the stack pointer.
   //
   // pthread_return: All the pthread_* functions return non-pointers,
   // except pthread_getspecific(), but it's ok: even though the
   // allocation/getting of the specifics pointer happens on the real CPU,
   // the set/get of the specific value is done in vg_libpthread.c on the
   // simd CPU, using the specifics pointer.
   //
   // The MALLOC request is also (unfortunately) lumped in with the pthread
   // ops... it does most certainly return a pointer, and one to a heap
   // block, which is marked as UNKNOWN.  Inaccurately marking it is not
   // really a problem, as the heap-blocks are entirely local to
   // vg_libpthread.c.
   //
   Seg seg = ( VG_(get_thread_archreg)(tid, reg) ? UNKNOWN : NONPTR );
   VG_(set_thread_shadow_archreg)( tid, reg, (UInt)seg );
}

static void post_reg_write_clientcall(ThreadId tid, UInt reg, Addr f )
{
   UInt      p;

   // Having to do this is a bit nasty...
   if (f == (Addr)SK_(malloc)
    || f == (Addr)SK_(__builtin_new)
    || f == (Addr)SK_(__builtin_vec_new)
    || f == (Addr)SK_(calloc)
    || f == (Addr)SK_(memalign)
    || f == (Addr)SK_(realloc)
   ) {
      // We remembered the last added segment;  make sure it's the right one.
      p = VG_(get_thread_archreg)(tid, reg);
      if ((UInt)NULL == p) {  // if alloc failed, eg. realloc on bogus pointer
         VG_(set_thread_shadow_archreg) ( tid, reg, (UInt)UNKNOWN );
      } else {
         sk_assert(p == Seg__a(last_seg_added));
         VG_(set_thread_shadow_archreg) ( tid, reg, (UInt)last_seg_added );
      }

   } else if (f == (Addr)SK_(free)
    || f == (Addr)SK_(__builtin_delete)
    || f == (Addr)SK_(__builtin_vec_delete)
    || f == (Addr)VG_(cli_block_size)
    || f == (Addr)VG_(message)
   ) {
      // Probably best to set the (non-existent!) return value to non-pointer.
      VG_(set_thread_shadow_archreg) ( tid, reg, (UInt)UNKNOWN );

   } else {
      // Anything else, probably best to set return value to non-pointer.
      //VG_(set_thread_shadow_archreg)(tid, reg, (UInt)UNKNOWN);
      Char fbuf[100];
      VG_(printf)("f = %p\n", f);
      VG_(get_fnname)(f, fbuf, 100);
      VG_(printf)("name = %s\n", fbuf);
      VG_(skin_panic)("argh: clientcall");
   }
}

/*--------------------------------------------------------------------*/
/*--- Sanity checking                                              ---*/
/*--------------------------------------------------------------------*/

/* Check that nobody has spuriously claimed that the first or last 16
   pages (64 KB) of address space have become accessible.  Failure of
   the following do not per se indicate an internal consistency
   problem, but they are so likely to that we really want to know
   about it if so. */
Bool SK_(cheap_sanity_check) ( void )
{
   if (IS_DISTINGUISHED_SM(primary_map[0])
       /* kludge: kernel drops a page up at top of address range for
          magic "optimized syscalls", so we can no longer check the
          highest page */
       /* && IS_DISTINGUISHED_SM(primary_map[65535]) */
      )
      return True;
   else
      return False;
}

Bool SK_(expensive_sanity_check) ( void )
{
   Int i;

   /* Make sure nobody changed the distinguished secondary. */
   for (i = 0; i < SEC_MAP_WORDS; i++)
      if (distinguished_secondary_map.vseg[i] != UNKNOWN)
         return False;

   return True;
}

/*--------------------------------------------------------------------*/
/*--- System calls                                                 ---*/
/*--------------------------------------------------------------------*/

void* SK_(pre_syscall) ( ThreadId tid, UInt syscallno, Bool is_blocking )
{
#if 0
   UInt mmap_flags;
   if (90 == syscallno) {
      // mmap: get contents of ebx to find args block, then extract 'flags'
      UInt* arg_block = (UInt*)VG_(get_thread_archreg)(tid, R_EBX);
      VG_(printf)("arg_block = %p\n", arg_block);
      mmap_flags = arg_block[3];
      VG_(printf)("flags = %d\n", mmap_flags);

   } else if (192 == syscallno) {
      // mmap2: get flags from 4th register arg
      mmap_flags = VG_(get_thread_archreg)(tid, R_ESI);

   } else {
      goto out;
   }

   if (0 != (mmap_flags & VKI_MAP_FIXED)) {
      VG_(skin_panic)("can't handle MAP_FIXED flag to mmap()");
   }

out:
#endif
   return NULL;
}

void SK_(post_syscall) ( ThreadId tid, UInt syscallno, void* pre_result,
                         Int res, Bool is_blocking )
{
   switch (syscallno) {

   // These ones don't return a pointer, so don't do anything -- already set
   // the segment to UNKNOWN in post_reg_write_default().
   //   1:           // exit              (never seen by skin)
   case 2 ... 16:    // read ... lchown
   //   17:          // break             (unimplemented)
   //   18:          // oldstat           (obsolete)
   case 19:          // lseek
   case 20 ... 25:   // getpid ... stime
   case 26:          // ptrace -- might return pointers, but not ones that
                     // deserve their own segment.
   case 27:          // alarm
   //   28:          // oldfstat          (obsolete)
   case 29 ... 30:   // pause ... utime
   //   31:          // stty              (unimplemented)
   //   32:          // gtty              (unimplemented)
   case 33 ... 34:   // access ... nice
   //   35:          // ftime             (obsolete and/or unimplemented)
   case 36 ... 43:   // sync ... times
   //   44:          // prof              (unimplemented)
   //   45:          // brk               (below)
   case 46 ... 47:   // setgid ... getgid
   //   48:          // signal            (never seen by skin)
   case 49 ... 50:   // geteuid ... getegid
   //   51:          // acct              (never seen by skin?  not handled by core)
   case 52:          // umount2
   //   53:          // lock              (unimplemented)
   case 54:          // ioctl -- assuming no pointers returned
   case 55:          // fcntl
   //   56:          // mpx               (unimplemented)
   case 57:          // setpgid
   //   58:          // ulimit            (unimplemented?  not handled by core)
   //   59:          // oldolduname       (obsolete)
   case 60 ... 67:   // sigaction
   //   68:          // sgetmask          (?? not handled by core)
   //   69:          // ssetmask          (?? not handled by core)
   case 70 ... 71:   // setreuid ... setguid
   //   72:          // sigsuspend        (handled by core? never seen by skins)
   case 73 ... 83:   // sigpending ... symlink
   //   84:          // oldlstat          (obsolete)
   case 85:          // readlink
   //   86:          // uselib            (not in core)
   //   87:          // swapon            (not in core)
   //   88:          // reboot            (not in core)
   //   89:          // readdir           (not in core)
   //   90:          // mmap              (below)
   case 91:          // munmap;  die_mem_munmap already called, segment removed
   case 92 ... 97:   // truncate ... setpriority
   //   98:          // profil            (not in core)
   case 99 ... 101:  // statfs ... ioperm
   case 102:         // socketcall -- assuming no pointers returned
   case 103 ... 108: // syslog ... fstat
   //   109:         // olduname          (obsolete)
   case 110 ... 111: // iopl ... vhangup
   //   112:         // idle              (not in core, only used by process 0)
   //   113:         // vm86old           (not in core)
   case 114:         // wait4
   //   115:         // swapoff           (not in core)
   case 116:         // sysinfo
   case 117:         // ipc -- assuming no pointers returned
   case 118:         // fsync
   //   119:         // sigreturn         (not in core, & shouldn't be called)
   //   120:         // clone             (not handled by core)
   //   121:         // setdomainname     (not in core)
   case 122 ... 126: // uname ... sigprocmask
   //   127:         // create_module     (not in core)
   case 128:         // init_module
   //   129:         // delete_module     (not in core)
   //   130:         // get_kernel_syms   (not in core)
   case 131 ... 133: // quotactl ... fchdir
   //   134:         // bdflush           (not in core)
   //   135:         // sysfs             (not in core)
   case 136:         // personality 
   //   137:         // afs_syscall       (not in core)
   case 138 ... 160: // setfsuid ... sched_get_priority_min
   //   161:         // rr_get_interval
   case 162:         // nanosleep
   //   163:         // mremap            (below)
   case 164 ... 165: // setresuid ... getresuid
   //   166:         // vm86              (not in core)
   //   167:         // query_module      (not in core) 
   case 168:         // poll
   //   169:         // nfsservctl        (not in core)
   case 170 ... 172: // setresgid ... prctl
   //   173:         // rt_sigreturn      (not in core)
   case 174 ... 177: // rt_sigaction ... rt_sigtimedwait
   //   178:         // rt_sigqueueinfo   (not in core)
   case 179 ... 191: // rt_sigsuspend ... ugetrlimit
   //   192:         // mmap              (below)
   case 193 ... 216: // truncate64 ... setfsgid32
   //   217:         // pivot_root        (not in core)
   //   218:         // mincore           (not in core)
   case 219 ... 221: // madvise ... fcntl64
   //   222:         // ???               (no such syscall?)
   //   223:         // security          (not in core)
   //   224:         // gettid            (not in core)
   //   225:         // readahead         (not in core)
   case 226 ... 237: // setxattr ... fremovexattr
   //   238:         // tkill             (not in core)
   case 239:         // sendfile64
   //   240:         // futex             (not in core)
   //   241:         // sched_setaffinity (not in core)
   //   242:         // sched_getaffinity (not in core)
   //   243:         // set_thread_area   (not in core)
   //   244:         // get_thread_area   (not in core)
   //   245:         // io_setup          (not in core)
   //   246:         // io_destroy        (not in core)
   //   247:         // io_getevents      (not in core)
   //   248:         // io_submit         (not in core)
   //   249:         // io_cancel         (not in core)
   //   250:         // alloc_hugepages   (not in core)
   //   251:         // free_hugepages    (not in core)
   //   252:         // exit_group        (never seen by skin)
   case 253:         // lookup_dcookie
   //   254:         // sys_epoll_create  (?)
   //   255:         // sys_epoll_ctl     (?)
   //   256:         // sys_epoll_wait    (?)
   //   257:         // remap_file_pages  (?)
   //   258:         // set_tid_address   (?)
      break;

   // With brk(), result (of kernel syscall, not glibc wrapper) is a heap
   // pointer.  Make the shadow UNKNOWN.
   case 45:
      VG_(set_return_from_syscall_shadow)( tid, (UInt)UNKNOWN );

   // With mmap, new_mem_mmap() has already been called and added the
   // segment (we did it there because we had the result address and size
   // handy).  So just set the return value shadow.
   case 90:    // mmap
   case 192:   // mmap2
      if (VG_(is_kerror)(res)) {
         // mmap() had an error, return value is a small negative integer
         VG_(set_return_from_syscall_shadow)( tid, (UInt)NONPTR );
      } else {
         // We remembered the last added segment;  make sure it's the right one.
         #if 0
         sk_assert(res == last_seg_added->left);
         VG_(set_return_from_syscall_shadow)( tid, (UInt)last_seg_added );
         #endif
         VG_(set_return_from_syscall_shadow)( tid, (UInt)UNKNOWN );
      }
      break;

   case 163:
      VG_(skin_panic)("can't handle mremap, sorry");
      break;

   default:
      VG_(printf)("syscallno == %d\n", syscallno);
      VG_(skin_panic)("unhandled syscall");
   }
}

/*--------------------------------------------------------------------*/
/*--- Functions called from generated code                         ---*/
/*--------------------------------------------------------------------*/

// XXX: could be more sophisticated -- actually track the lowest/highest
// valid address used by the program, and then return False for anything
// below that (using a suitable safety margin).  Also, nothing above
// 0xc0000000 is valid [unless you've changed that in your kernel]
static __inline__ Bool looks_like_a_pointer(Addr a)
{
   return (a > 0x01000000 && a < 0xff000000);
}

static __inline__ __attribute__((regparm(1)))
Seg nonptr_or_unknown(UInt x)
{
   return ( looks_like_a_pointer(x) ? UNKNOWN : NONPTR );
}

static __attribute__((regparm(1)))
void print_BB_entry(UInt bb)
{
   VG_(printf)("%u =\n", bb);
}

// This function is called *a lot*; inlining it sped up Konqueror by 20%.
static __inline__
void check_load_or_store(Bool is_write, Addr m, UInt sz, Seg mptr_vseg)
{
   VGP_PUSHCC(VgpCheckLoadStore);

   if (UNKNOWN == mptr_vseg) {
      // do nothing

   } else if (BOTTOM == mptr_vseg) {
      // do nothing

   } else if (NONPTR == mptr_vseg) {
      record_loadstore_error( m, sz, mptr_vseg, is_write );

   } else {
      // check all segment ranges in the circle
      // if none match, warn about 1st seg
      // else,          check matching one isn't freed
      Bool is_ok = False;
      Seg  curr  = mptr_vseg;
      Addr mhi;

      // Accesses partly outside range are an error, unless it's an aligned
      // word-sized read, and --partial-loads-ok=yes.  This is to cope with
      // gcc's/glibc's habits of doing word-sized accesses that read past
      // the ends of arrays/strings.
      if (!is_write && 4 == sz && clo_partial_loads_ok && IS_ALIGNED4(m))
         mhi = m;
      else
         mhi = m+sz-1;

      #if 0
      while (True) {
         lo  = curr->data;
         lim = lo + curr->size;
         if (m >= lo && mlim <= lim) { is_ok = True; break; }
         curr = curr->links;
         if (curr == mptr_vseg)      {               break; }
      }
      #else
      // This version doesn't do the link-segment chasing
      is_ok = Seg__containsI(curr, m, mhi);
      #endif

      // If it's an overrun/underrun of a freed block, don't give both
      // warnings, since the first one mentions that the block has been
      // freed.
      if ( ! is_ok || Seg__is_freed(curr) )
         record_loadstore_error( m, sz, mptr_vseg, is_write );
   }

   VGP_POPCC(VgpCheckLoadStore);
}

// return m.vseg
static __attribute__((regparm(2)))
Seg check_load4(Addr m, Seg mptr_vseg)
{
   Seg vseg;
   VGP_PUSHCC(VgpCheckLoad4);
   check_load_or_store(/*is_write*/False, m, 4, mptr_vseg);
   if (IS_ALIGNED4(m)) {
      vseg = get_mem_vseg(m);
   } else {
      vseg = nonptr_or_unknown( *(UInt*)m );
   }
   VGP_POPCC(VgpCheckLoad4);
   return vseg;
}

static __attribute__((regparm(2)))
Seg check_load2(Addr m, Seg mptr_vseg)
{
   VGP_PUSHCC(VgpCheckLoad21);
   check_load_or_store(/*is_write*/False, m, 2, mptr_vseg);
   VGP_POPCC(VgpCheckLoad21);
   return NONPTR;
}

static __attribute__((regparm(2)))
Seg check_load1(Addr m, Seg mptr_vseg)
{
   VGP_PUSHCC(VgpCheckLoad21);
   check_load_or_store(/*is_write*/False, m, 1, mptr_vseg);
   VGP_POPCC(VgpCheckLoad21);
   return NONPTR;
}


static Seg check_store4___t_vseg;

static __attribute__((regparm(3)))
void check_store4(Addr m, Seg mptr_vseg, UInt t)
{
   VGP_PUSHCC(VgpCheckStore4);
   check_load_or_store(/*is_write*/True, m, 4, mptr_vseg);
   // Actually *do* the STORE here
   *(UInt*)m = t;
   if (IS_ALIGNED4(m)) {
      set_mem_vseg( m, check_store4___t_vseg );
   } else {
      // straddling two words
      m = VG_ROUNDDN(m,4);
      set_mem_vseg( m, nonptr_or_unknown( *(UInt*)m ) );
      m += 4;
      set_mem_vseg( m, nonptr_or_unknown( *(UInt*)m ) );
   }
   VGP_POPCC(VgpCheckStore4);
}

static __attribute__((regparm(3)))
void check_store2(Addr m, Seg mptr_vseg, UInt t)
{
   VGP_PUSHCC(VgpCheckStore21);
   check_load_or_store(/*is_write*/True, m, 2, mptr_vseg);
   // Actually *do* the STORE here  (Nb: cast must be to 2-byte type!)
   *(UShort*)m = t;
   if (! IS_ALIGNED4(m+1)) {
      // within one word
      m = VG_ROUNDDN(m,4);
      set_mem_vseg( m, nonptr_or_unknown( *(UInt*)m ) );
   } else {
      // straddling two words
      m = VG_ROUNDDN(m,4);
      set_mem_vseg( m, nonptr_or_unknown( *(UInt*)m ) );
      m += 4;
      set_mem_vseg( m, nonptr_or_unknown( *(UInt*)m ) );
   }
   VGP_POPCC(VgpCheckStore21);
}

static __attribute__((regparm(3)))
void check_store1(Addr m, Seg mptr_vseg, UInt t)
{
   VGP_PUSHCC(VgpCheckStore21);
   check_load_or_store(/*is_write*/True, m, 1, mptr_vseg);
   // Actually *do* the STORE here  (Nb: cast must be to 1-byte type!)
   *(UChar*)m = t;
   m = VG_ROUNDDN(m,4);
   set_mem_vseg( m, nonptr_or_unknown( *(UInt*)m ) );
   VGP_POPCC(VgpCheckStore21);
}


#define CHECK_FPU_R(N) \
static __attribute__((regparm(2))) \
void check_fpu_r##N(Addr m, Seg mptr_vseg) \
{ \
   VGP_PUSHCC(VgpCheckFpuR); \
   check_load_or_store(/*is_write*/False, m, N, mptr_vseg); \
   VGP_POPCC(VgpCheckFpuR); \
}
CHECK_FPU_R(2);
CHECK_FPU_R(4);
CHECK_FPU_R(8);
CHECK_FPU_R(10);
CHECK_FPU_R(16);
CHECK_FPU_R(28);
CHECK_FPU_R(108);


#define CHECK_FPU_W(N) \
static __attribute__((regparm(2))) \
void check_fpu_w##N(Addr m, Seg mptr_vseg) \
{ \
   VGP_PUSHCC(VgpCheckFpuW); \
   check_load_or_store(/*is_write*/True, m, N, mptr_vseg); \
   set_mem_nonptr( m, N ); \
   VGP_POPCC(VgpCheckFpuW); \
}
CHECK_FPU_W(2);
CHECK_FPU_W(4);
CHECK_FPU_W(8);
CHECK_FPU_W(10);
CHECK_FPU_W(16);
CHECK_FPU_W(28);
CHECK_FPU_W(108);


// Nb: if the result is BOTTOM, return immedately -- don't let BOTTOM
//     be changed to NONPTR by a range check on the result.
#define BINOP(bt, nn, nu, np, un, uu, up, pn, pu, pp) \
   if (BOTTOM == seg1 || BOTTOM == seg2) { bt; \
   } else if (NONPTR == seg1)  { if      (NONPTR == seg2)  { nn; } \
                                 else if (UNKNOWN == seg2) { nu; } \
                                 else                      { np; } \
   } else if (UNKNOWN == seg1) { if      (NONPTR == seg2)  { un; } \
                                 else if (UNKNOWN == seg2) { uu; } \
                                 else                      { up; } \
   } else                      { if      (NONPTR == seg2)  { pn; } \
                                 else if (UNKNOWN == seg2) { pu; } \
                                 else                      { pp; } \
   }

#define BINERROR(opc) \
   record_arith_error(0xA5, 0xA5, seg1, seg2, opc); \
   out = NONPTR

// Extra arg for result of various ops
static UInt do_op___result;

// -------------
//  + | n  ?  p
// -------------
//  n | n  ?  p
//  ? | ?  ?  ?
//  p | p  ?  e   (all results become n if they look like a non-pointer)
// -------------
static Seg do_add4_result(Seg seg1, Seg seg2, Opcode opcode)
{
   Seg out;

   BINOP(
      return BOTTOM,
      out = NONPTR,  out = UNKNOWN, out = seg2,
      out = UNKNOWN, out = UNKNOWN, out = UNKNOWN,
      out = seg1,    out = UNKNOWN,       BINERROR(opcode)
   );
   return ( looks_like_a_pointer(do_op___result) ? out : NONPTR );
}

static __attribute__((regparm(2)))
Seg do_add4(Seg seg1, Seg seg2)
{
   Seg out;

   VGP_PUSHCC(VgpDoAdd);
   out = do_add4_result(seg1, seg2, ADD);
   VGP_POPCC (VgpDoAdd);
   return out;
}

// -------------
//  - | n  ?  p      (Nb: operation is seg1 - seg2)
// -------------
//  n | n  ?  n+     (+) happens a lot due to "cmp", but result should never
//  ? | ?  ?  n/B        be used, so give 'n'
//  p | p  p? n*/B   (*) and possibly link the segments
// -------------
static __attribute__((regparm(2)))
Seg do_sub4(Seg seg1, Seg seg2)
{
   Seg out;

   VGP_PUSHCC(VgpDoSub);
   // Nb: when returning BOTTOM, don't let it go through the range-check;
   //     a segment linking offset can easily look like a nonptr.
   BINOP(
      VGP_POPCC(VgpDoSub); return BOTTOM,
      out = NONPTR,  out = UNKNOWN,    out = NONPTR,
      out = UNKNOWN, out = UNKNOWN,    VGP_POPCC(VgpDoSub); return BOTTOM,
      out = seg1,    out = seg1/*??*/, VGP_POPCC(VgpDoSub); return BOTTOM
   );
   #if 0
         // This is for the p-p segment-linking case
         Seg end2 = seg2;
         while (end2->links != seg2) end2 = end2->links;
         end2->links = seg1->links;
         seg1->links = seg2;
         return NONPTR;
   #endif
   VGP_POPCC (VgpDoSub);
   return ( looks_like_a_pointer(do_op___result) ? out : NONPTR );
}

// -------------
// +-.| n  ?  p   (Nb: not very certain about these ones)
// -------------
//  n | n  ?  e
//  ? | ?  ?  e
//  p | p  e  e
// -------------
static __inline__
Seg do_adcsbb4(Seg seg1, Seg seg2, Opcode opc)
{
   Seg out;

   VGP_PUSHCC(VgpDoAdcSbb);
   BINOP(
      VGP_POPCC(VgpDoAdcSbb); return BOTTOM,
      out = NONPTR,  out = UNKNOWN, BINERROR(opc),
      out = UNKNOWN, out = UNKNOWN, BINERROR(opc),
      out = seg1,    BINERROR(opc), BINERROR(opc)
   );
   VGP_POPCC(VgpDoAdcSbb);
   return ( looks_like_a_pointer(do_op___result) ? out : NONPTR );
}

static __attribute__((regparm(2)))
Seg do_adc4(Seg seg1, Seg seg2)
{
   return do_adcsbb4(seg1, seg2, ADC);
}

static __attribute__((regparm(2)))
Seg do_sbb4(Seg seg1, Seg seg2)
{
   return do_adcsbb4(seg1, seg2, SBB);
}

// -------------
//  & | n  ?  p
// -------------
//  n | n  ?  p
//  ? | ?  ?  ?
//  p | p  ?  *  (*) if p1==p2 then p else e
// -------------
static UInt do_and4___args_diff;

static __attribute__((regparm(2)))
Seg do_and4(Seg seg1, Seg seg2)
{
   Seg out;

   VGP_PUSHCC(VgpDoAnd);
   if (0 == do_and4___args_diff) {
      // p1==p2
      out = seg1;
   } else {
      BINOP(
         VGP_POPCC(VgpDoAnd); return BOTTOM,
         out = NONPTR,  out = UNKNOWN, out = seg2,
         out = UNKNOWN, out = UNKNOWN, out = UNKNOWN,
         out = seg1,    out = UNKNOWN,       BINERROR(AND)
      );
   }
   out = ( looks_like_a_pointer(do_op___result) ? out : NONPTR );
   VGP_POPCC (VgpDoAnd);
   return out;
}

// -------------
// `|`| n  ?  p
// -------------
//  n | n  ?  p
//  ? | ?  ?  ?
//  p | p  ?  e
// -------------
static __attribute__((regparm(2)))
Seg do_or4(Seg seg1, Seg seg2)
{
   Seg out;

   VGP_PUSHCC(VgpDoOr);
   BINOP(
      VGP_POPCC(VgpDoOr); return BOTTOM,
      out = NONPTR,  out = UNKNOWN, out = seg2,
      out = UNKNOWN, out = UNKNOWN, out = UNKNOWN,
      out = seg1,    out = UNKNOWN,       BINERROR(OR)
   );
   out = ( looks_like_a_pointer(do_op___result) ? out : NONPTR );
   VGP_POPCC (VgpDoOr);
   return out;
}

// -------------
// L1 | n  ?  p
// -------------
//  n | n  ?  p
//  ? | ?  ?  p
// -------------
static Seg do_lea(Bool k_looks_like_a_ptr, Seg seg1)
{
   Seg out;

   if (BOTTOM == seg1) return BOTTOM;

   if (k_looks_like_a_ptr && NONPTR == seg1)
      out = UNKNOWN;      // ?(n)
   else
      out = seg1;         // n(n), n(?), n(p), ?(?), ?(p)

   return ( looks_like_a_pointer(do_op___result) ? out : NONPTR );
}

static __attribute__((regparm(1)))
Seg do_lea1_unknown(Seg seg1)
{
   Seg out;

   VGP_PUSHCC(VgpDoLea1);
   out = do_lea( /*k_looks_like_a_ptr*/True, seg1 );
   VGP_POPCC (VgpDoLea1);
   return out;
}

static UInt do_lea2___k;

static __attribute__((regparm(2)))
Seg do_lea2_1(Seg seg1, Seg seg2)
{
   Seg out;

   // a) Combine seg1 and seg2 as for ADD, giving a result.
   // b) Combine that result with k as for LEA1.
   VGP_PUSHCC(VgpDoLea2);
   out = do_lea( looks_like_a_pointer(do_lea2___k),
                 do_add4_result(seg1, seg2, LEA2) );
   VGP_POPCC (VgpDoLea2);
   return out;
}

static __attribute__((regparm(2)))
Seg do_lea2_n(Seg seg1, Seg seg2)
{
   VGP_PUSHCC(VgpDoLea2);
   VGP_POPCC (VgpDoLea2);

   if (is_known_segment(seg2)) {
      // Could do with AsmError
      VG_(message)(Vg_UserMsg,
                   "\nScaling known pointer by value > 1 in lea instruction");
   }
   return do_lea(looks_like_a_pointer(do_lea2___k), seg1);
}

// -------------
//  - | n  ?  p
// -------------
//    | n  n  n
// -------------
static __attribute__((regparm(2)))
Seg do_neg4(Seg seg1, UInt result)
{
   if (BOTTOM == seg1) return BOTTOM;

   return NONPTR;
}

// -------------
//  ~ | n  ?  p
// -------------
//    | n  n  n
// -------------
static __attribute__((regparm(2)))
Seg do_not4(Seg seg1, UInt result)
{
   if (BOTTOM == seg1) return BOTTOM;

   return NONPTR;
}

// Pointers are rarely multiplied, but sometimes legitimately, eg. as hash
// function inputs.  But two pointers args --> error.
static __attribute__((regparm(2)))
void check_mul4(Seg seg1, Seg seg2)
{
   if (is_known_segment(seg1) && is_known_segment(seg2))
      record_arith_error(0xA5, 0xA5, seg1, seg2, -VGOFF_(helper_mul_32_64));
}

static __attribute__((regparm(2)))
void check_imul4(Seg seg1, Seg seg2)
{
   if (is_known_segment(seg1) && is_known_segment(seg2))
      record_arith_error(0xA5, 0xA5, seg1, seg2, -VGOFF_(helper_imul_32_64));
}

// seg1 / seg2: div_64_32 can take a pointer for its 2nd arg (seg1).
static __attribute__((regparm(2)))
void check_div4(Seg seg1, Seg seg2)
{
   if (is_known_segment(seg2))
      record_arith_error(0xA5, 0xA5, seg1, seg2, -VGOFF_(helper_div_64_32));
}

// seg1 / seg2: idiv_64_32 can take a pointer for its 2nd or 3rd arg
// (because the 3rd arg, ie. the high-word, will be derived from the
// low-word with a 'sar' in order to get the sign-bit correct).  But we only
// check the lo-word.
static __attribute__((regparm(2)))
void check_idiv4(Seg seg1, Seg seg2)
{
   if (is_known_segment(seg2))
      record_arith_error(0xA5, 0xA5, seg1, seg2, -VGOFF_(helper_idiv_64_32));
}


/*--------------------------------------------------------------------*/
/*--- Instrumentation                                              ---*/
/*--------------------------------------------------------------------*/

static void set_nonptr_or_unknown(UCodeBlock* cb, UInt t)
{
   sk_assert(0 == t % 2);     // check number is even, ie. a normal tempreg
   VG_(ccall_R_R)(cb, (Addr)nonptr_or_unknown,
                  t,            // t      (reg)
                  SHADOW(t),    // t.vseg (reg)(retval)
                  1);
}

static void set_shadowreg(UCodeBlock* cb, UInt q, Seg seg)
{
   sk_assert(1 == q % 2);     // check number is odd, ie. a shadow tempreg
   VG_(lit_to_reg)(cb, (UInt)seg, q);
}

static void set_nonptr(UCodeBlock* cb, UInt q)
{
   set_shadowreg(cb, q, NONPTR);
}

__attribute__((unused))
static void record_binary_arith_args(UCodeBlock* cb, UInt arg1, UInt arg2)
{
   VG_(reg_to_globvar)(cb, arg1, &actual_arg1);
   VG_(reg_to_globvar)(cb, arg2, &actual_arg2);
}

//-----------------------------------------------------------------------
// Approach taken for range-checking for NONPTR/UNKNOWN-ness as follows.
//
// Range check (NONPTR/seg): 
// - after modifying a word-sized value in/into a TempReg:
//    - {ADD, SUB, ADC, SBB, AND, OR, XOR, LEA, LEA2, NEG, NOT}L
//    - BSWAP
// 
// Range check (NONPTR/UNKNOWN):
// - when introducing a new word-sized value into a TempReg:
//    - MOVL l, t2
//
// - when copying a word-sized value which lacks a corresponding segment
//   into a TempReg:
//    - straddled LDL
//
// - when a sub-word of a word (or two) is updated:
//    - SHROTL
//    - {ADD, SUB, ADC, SBB, AND, OR, XOR, SHROT, NEG, NOT}[WB]
//    - PUT[WB]
//    - straddled   STL (2 range checks)
//    - straddled   STW (2 range checks)
//    - unstraddled STW
//    - STB
//    
// Just copy:
// - when copying word-sized values:
//    - MOVL t1, t2 (--optimise=no only)
//    - CMOV
//    - GETL, PUTL
//    - unstraddled LDL, unstraddled STL
//
// - when barely changing
//    - INC[LWB]/DEC[LWB]
// 
// Set to NONPTR:
// - after copying a sub-word value into a TempReg:
//    - MOV[WB] l, t2
//    - GET[WB]
//    - unstraddled LDW
//    - straddled   LDW
//    - LDB
//    - POP[WB]
//
// - after copying an obvious non-ptr into a TempReg:
//    - GETF
//    - CC2VAL
//    - POPL
//
// - after copying an obvious non-ptr into a memory word:
//    - FPU_W
// 
// Do nothing:
// - LOCK, INCEIP
// - WIDEN[WB]
// - JMP, JIFZ
// - CALLM_[SE], PUSHL, CALLM, CLEAR
// - FPU, FPU_R (and similar MMX/SSE ones)
//
UCodeBlock* SK_(instrument)(UCodeBlock* cb_in, Addr orig_addr)
{
   static UInt bb = 0;

   UCodeBlock* cb;
   Int         i;
   UInstr*     u;
   Addr        helper;
   UInt        callm_shadow_arg4v[3];
   UInt        callm_arg4c = -1;

   cb = VG_(setup_UCodeBlock)(cb_in);

   // Print BB number upon execution
   if (0)
      VG_(ccall_L_0)(cb, (Addr)print_BB_entry, bb++, 1);


   for (i = 0; i < VG_(get_num_instrs)(cb_in); i++) {
      u = VG_(get_instr)(cb_in, i);

      //-- Start main switch -------------------------------------------
      switch (u->opcode) {

      case NOP:
         break;

      case LOCK:
      case INCEIP:
         VG_(copy_UInstr)(cb, u);
         break;

      //--- Value movers -----------------------------------------------
      case MOV:
         // a) Do MOV
         VG_(copy_UInstr)(cb, u);
         if (4 == u->size) {
            if (Literal == u->tag1) {
               // MOVL l, t2
               // b) t2.vseg = NONPTR/UNKNOWN
               set_shadowreg(cb, SHADOW(u->val2), nonptr_or_unknown(u->lit32));
            } else {
               // MOVL t1, t2 (only occurs if --optimise=no)
               // b) t2.vseg = t1.vseg
               uInstr2(cb, MOV, 4, TempReg, SHADOW(u->val1),
                                   TempReg, SHADOW(u->val2));
            }
         } else {
            // MOV[WB] l, t2
            // b) t2.vseg = NONPTR
            sk_assert(Literal == u->tag1);
            set_nonptr(cb, SHADOW(u->val2));
         }
         break;

      case CMOV:
         // CMOV t1, t2
         // a) t2.vseg = t1.vseg, if condition holds
         // b) Do CMOV
         sk_assert(4 == u->size);
         uInstr2(cb, CMOV, 4, TempReg, SHADOW(u->val1),
                              TempReg, SHADOW(u->val2));
         uCond(cb, u->cond);
         uFlagsRWU(cb, u->flags_r, u->flags_w, FlagsEmpty);
         VG_(copy_UInstr)(cb, u);
         break;

      case GET:
         // a) Do GET
         VG_(copy_UInstr)(cb, u);
         if (4 == u->size) {
            // GETL r, t2
            // b) t2.vseg = r.vseg
            uInstr2(cb, GETV, 4, ArchReg, u->val1, TempReg, SHADOW(u->val2));
         } else {
            // GET[WB] r, t2
            // b) t2.vseg = NONPTR
            set_nonptr(cb, SHADOW(u->val2));
         }
         break;

      case PUT: {
         // a) Do PUT
         VG_(copy_UInstr)(cb, u);
         if (4 == u->size) {
            // PUTL t1, r
            // b) r.vseg = t1.vseg
            uInstr2(cb, PUTV, 4, TempReg, SHADOW(u->val1), ArchReg, u->val2);
         } else {
            // PUT[WB] t1, r
            // b) r.vseg = NONPTR/UNKNOWN
            //    (GET the result of the PUT[WB], look at the resulting
            //     word, PUTV the shadow result.  Ugh.)
            UInt t_res = newTemp(cb);
            uInstr2(cb, GET,  4, ArchReg, u->val2, TempReg, t_res);
            set_nonptr_or_unknown(cb, t_res);
            uInstr2(cb, PUTV, 4, TempReg, SHADOW(t_res), ArchReg, u->val2);
         }
         break;
      }

      case GETF:
         // GETF t1
         // a) t1.vseg = NONPTR
         // b) Do GETF
         set_nonptr(cb, SHADOW(u->val1));
         VG_(copy_UInstr)(cb, u);
         break;

      case PUTF:
         // PUTF t1
         // a) Do PUTF
         VG_(copy_UInstr)(cb, u);
         break;

      case LOAD:
         // LD[LWB] m, t
         if      (4 == u->size) helper = (Addr)check_load4;
         else if (2 == u->size) helper = (Addr)check_load2;
         else if (1 == u->size) helper = (Addr)check_load1;
         else    VG_(skin_panic)("bad LOAD size");

         // a) Check segments match (in helper)
         // b) t.vseg = m.vseg (helper returns m.vseg)
         // c) Do LOAD (must come after segment check!)
         VG_(ccall_RR_R)(cb, helper,
                         u->val1,            // m              (reg)
                         SHADOW(u->val1),    // m_ptr.vseg     (reg)
                         SHADOW(u->val2),    // t.vseg (retval)(reg)
                         2);
         VG_(copy_UInstr)(cb, u);
         break;

      case STORE:
         if (4 == u->size) {
            // Put t.vseg in globvar-arg
            VG_(reg_to_globvar)(cb, SHADOW(u->val1),
                                (UInt*) & check_store4___t_vseg);
            helper = (Addr)check_store4;
         } 
         else if (1 == u->size) helper = (Addr)check_store1;
         else if (2 == u->size) helper = (Addr)check_store2;
         else                   VG_(skin_panic)("bad size for STORE");
            
         // a) Check segments match
         // b) Do STORE (after segment check; done by helper)
         //
         // STL t, m
         // c) if aligned(m), m.vseg = t.vseg
         //    else vseg of both words touched set to UNKNOWN
         //
         // ST[WB] t, m
         // c) if non-straddling(m), m.vseg = NONPTR/UNKNOWN
         //    else vseg of both words touched set to UNKNOWN
         VG_(ccall_RRR_0)(cb, helper,
                              u->val2,          // m          (reg)
                              SHADOW(u->val2),  // m_ptr.vseg (reg)
                              u->val1,          // t          (reg)
                              3);
         break;

      //--- Binary arithmetic ------------------------------------------
      case ADD: helper = (Addr)do_add4;  goto do_add_sub;
      case ADC: helper = (Addr)do_adc4;  goto do_add_sub;
      case SUB: helper = (Addr)do_sub4;  goto do_add_sub;
      case SBB: helper = (Addr)do_sbb4;  goto do_add_sub;
       do_add_sub:
         // a) Do OP (and record result in globvar)
//record_binary_arith_args(cb, u->val2, t1);

         VG_(copy_UInstr)(cb, u);
         if (4 == u->size) {
            UInt t1;
            if (Literal == u->tag1) {
               // OPL l, t2
               Seg segL = nonptr_or_unknown(u->lit32);
               t1 = newTemp(cb);
               VG_(lit_to_reg)(cb, u->lit32,   t1);
               VG_(lit_to_reg)(cb, (UInt)segL, SHADOW(t1));
            } else if (ArchReg == u->tag1) {
               // OPL r, t2
               t1 = newTemp(cb);
               uInstr2(cb, GET,  4, ArchReg, u->val1, TempReg, t1);
               uInstr2(cb, GETV, 4, ArchReg, u->val1, TempReg, SHADOW(t1));
            } else {
               // OPL t1, t2
               sk_assert(TempReg == u->tag1);
               t1 = u->val1;
            }

            // Put result in globvar-arg
            VG_(reg_to_globvar)(cb, u->val2, &do_op___result);
            // b) Check args (if necessary;  ok to do after the OP itself)
            // c) Update t.vseg
            VG_(ccall_RR_R)(cb, helper,
                            SHADOW(u->val2),    // t2.vseg (reg)
                            SHADOW(t1),         // t1.vseg (reg)
                            SHADOW(u->val2),    // t2.vseg (reg)(retval)
                            2);
         } else {
            // OP[WB] x, t2
            // b) t2.vseg = UKNOWN
            set_nonptr_or_unknown(cb, u->val2);
         }
         break;

      case AND:
         // a) Do AND
         VG_(copy_UInstr)(cb, u);
         if (4 == u->size) {
            // ANDL t1, t2
            // Find difference between t1 and t2 (to determine if they're equal)
            // put in globvar-arg
            UInt t_equal = newTemp(cb);
            uInstr2(cb, MOV, 4, TempReg, u->val2, TempReg, t_equal);
            uInstr2(cb, SUB, 4, TempReg, u->val1, TempReg, t_equal);
            VG_(reg_to_globvar)(cb, t_equal, &do_and4___args_diff);
            // Put result in globvar-arg
            VG_(reg_to_globvar)(cb, u->val2, &do_op___result);
            // b) Update t2.vseg
            VG_(ccall_RR_R)(cb, (Addr)do_and4,
                           SHADOW(u->val1),    // t1.vseg (reg)
                           SHADOW(u->val2),    // t2.vseg (reg)
                           SHADOW(u->val2),    // t2.vseg (reg)(retval)
                           2);
         } else {
            // AND[WB] t1, t2
            // b) t2.vseg = NONPTR/UNKNOWN
            set_nonptr_or_unknown(cb, u->val2);
         }
         break;

      case OR:
         // a) Do OR
         VG_(copy_UInstr)(cb, u);
         if (4 == u->size) {
            // ORL t1, t2
            // Put result in globvar-arg
            VG_(reg_to_globvar)(cb, u->val2, &do_op___result);
            // b) Update t2.vseg
            VG_(ccall_RR_R)(cb, (Addr)do_or4,
                           SHADOW(u->val1),    // t1.vseg (reg)
                           SHADOW(u->val2),    // t2.vseg (reg)
                           SHADOW(u->val2),    // t2.vseg (reg)(retval)
                           2);
         } else {
            // OR[WB] t1, t2
            // b) t2.vseg = NONPTR/UNKNOWN
            set_nonptr_or_unknown(cb, u->val2);
         }
         break;

      // With XOR, the result is likely to be a nonptr, but could
      // occasionally not be due to weird things like xor'ing a pointer with
      // zero, or using the xor trick for swapping two variables.  So,
      // simplest thing is to just look at the range.
      case XOR:
         // XOR[LWB] x, t2
         // a) Do XOR
         // b) t2.vseg = NONPTR/UNKNOWN
         VG_(copy_UInstr)(cb, u);
         set_nonptr_or_unknown(cb, u->val2);
         break;

      // Nb: t1 is always 1-byte, and thus never a pointer.
      case SHL: case SHR: case SAR:
      case ROL: case ROR:
         // SHROT x, t2
         // a) Do SHROT
         // b) t2.vseg = NONPTR/UNKNOWN
         VG_(copy_UInstr)(cb, u);
         set_nonptr_or_unknown(cb, u->val2);
         break;

      // LEA2 t1,t2,t3:  t3 = lit32 + t1 + (t2 * extra4b)
      // Used for pointer computations, and for normal arithmetic (eg.
      // "lea (%r1,%r1,4),%r2" multiplies by 5).
      case LEA2:
         sk_assert(4 == u->size);
         // a) Do LEA2
         VG_(copy_UInstr)(cb, u);
         if (1 == u->extra4b) {
            // t3 = lit32 + t1 + (t2*1)  (t2 can be pointer)
            helper = (Addr)do_lea2_1;
         } else {
            // t3 = lit32 + t1 + (t2*n)  (t2 cannot be pointer)
            helper = (Addr)do_lea2_n;
         }
         // Put k in globvar-arg
         VG_(lit_to_globvar)(cb, u->lit32, &do_lea2___k);
         // Put result in globvar-arg
         VG_(reg_to_globvar)(cb, u->val3,  &do_op___result);
         // b) Check args
         // c) Update t3.vseg
         VG_(ccall_RR_R)(cb, helper,
                         SHADOW(u->val1),    // t1.vseg (reg)
                         SHADOW(u->val2),    // t2.vseg (reg)
                         SHADOW(u->val3),    // t3.vseg (reg)(retval)
                         2);
         break;

      //--- Unary arithmetic -------------------------------------------
      case NEG: helper = (Addr)do_neg4;  goto do_neg_not;
      case NOT: helper = (Addr)do_not4;  goto do_neg_not;
       do_neg_not:
         // a) Do NEG/NOT
         VG_(copy_UInstr)(cb, u);
         if (4 == u->size) {
            // NEGL/NOTL t1
            // b) Check args (NEG only)
            // c) Update t1.vseg
            VG_(ccall_RR_R)(cb, helper,
                            SHADOW(u->val1),    // t1.vseg (reg)
                            u->val2,            // t2      (reg)
                            SHADOW(u->val1),    // t1.vseg (reg)(retval)
                            2);
         } else {
            // NEG[WB]/NOT[WB] t1
            // b) Update t1.vseg
            set_nonptr_or_unknown(cb, u->val2);
         }
         break;

      case INC:
      case DEC:
         // INC[LWB]/DEC[LWB] t1
         // a) Do INC/DEC
         // b) t1.vseg unchanged (do nothing)
         VG_(copy_UInstr)(cb, u);
         break;

      case LEA1:
         sk_assert(4 == u->size);
         // LEA1 k(t1), t2
         // b) Do LEA1
         VG_(copy_UInstr)(cb, u);
         if ( ! looks_like_a_pointer(u->lit32) ) {
            // a) simple, 99.9% case: k is a known nonptr, t2.vseg = t1.vseg
            uInstr2(cb, MOV, 4, TempReg, SHADOW(u->val1),
                                TempReg, SHADOW(u->val2));
         } else {
            // Put result in globvar-arg (not strictly necessary, because we
            // have a spare CCALL slot, but do it for consistency)
            VG_(reg_to_globvar)(cb, u->val2,  &do_op___result);
            // a) complicated, 0.1% case: k could be a pointer
            VG_(ccall_R_R)(cb, (Addr)do_lea1_unknown,
                           SHADOW(u->val1),    // t1.vseg (reg)
                           SHADOW(u->val2),    // t2.vseg (reg)(retval)
                           1);
         }
         break;

      //--- End of arithmetic ------------------------------------------

      case WIDEN:
         // WIDEN t1
         // a) Do WIDEN
         VG_(copy_UInstr)(cb, u);
         break;

      case CC2VAL:
         // CC2VAL t1
         // a) t1.vseg = NONPTR
         // b) Do CC2VAL
         set_nonptr(cb, SHADOW(u->val1));
         VG_(copy_UInstr)(cb, u);
         break;

      case BSWAP:
         // BSWAP t1
         // a) t1.vseg = UNKNOWN
         // b) Do BSWAP
         set_nonptr_or_unknown(cb, u->val1);
         VG_(copy_UInstr)(cb, u);
         break;

      case JIFZ:
      case JMP:
         VG_(copy_UInstr)(cb, u);
         break;

      //--- CALLM and related ------------------------------------------
      // Basic form of CALLMs, as regexp:
      //   CALLM_S (GET? PUSH)* CALLM (POP PUT? | CLEAR)* CALLM_E
      //
      // How we're doing things:
      // - PUSH:  Args will be popped/cleared, so no need to set the shadows.
      //          Check input depending on the callee.
      // - CALLM: Do nothing, already checked inputs on PUSH.
      // - POP:   Depending on the callee, set popped outputs as
      //          pointer/non-pointer.  But all callees seem to produce
      //          NONPTR outputs, so just set output arg so straight off.
      case CALLM_S:
         sk_assert(-1 == callm_arg4c);
         callm_arg4c = 0;
         break;

      case CALLM_E:
         sk_assert(-1 != callm_arg4c);
         callm_arg4c = -1;
         break;

      case PUSH:
         sk_assert(-1 != callm_arg4c);
         if (4 == u->size) {
            // PUSHL t1
            callm_shadow_arg4v[ callm_arg4c++ ] = SHADOW(u->val1); 
         }
         // a) Do PUSH
         VG_(copy_UInstr)(cb, u);
         break;

      // All the ops using helpers certainly don't return pointers, except
      // possibly, sh[rl]dl.  I'm not sure about them, but make the result
      // NONPTR anyway.
      case POP:
         // POP t1
         // a) set t1.vseg == NONPTR
         // b) Do POP
         sk_assert(-1 != callm_arg4c);
         set_nonptr(cb, SHADOW(u->val1));
         VG_(copy_UInstr)(cb, u);
         break;

      case CLEAR:
         sk_assert(-1 != callm_arg4c);
         VG_(copy_UInstr)(cb, u);
         break;

      case CALLM:
         if (!(0 <= callm_arg4c && callm_arg4c <= 4)) {
            VG_(printf)("callm_arg4c = %d\n", callm_arg4c);
            VG_(skin_panic)("bleh");
         }

         #define V(name)   (VGOFF_(helper_##name) == u->val1)

         if (V(mul_32_64)) {
            sk_assert(2 == callm_arg4c);
            VG_(ccall_RR_0)(cb, (Addr)check_mul4,
                            callm_shadow_arg4v[1], // arg2.vseg (reg)
                            callm_shadow_arg4v[0], // arg1.vseg (reg)
                            2);

         } else if (V(imul_32_64)) {
            sk_assert(2 == callm_arg4c);
            VG_(ccall_RR_0)(cb, (Addr)check_imul4,
                            callm_shadow_arg4v[1], // arg2.vseg (reg)
                            callm_shadow_arg4v[0], // arg1.vseg (reg)
                            2);

         } else if (V(div_64_32)) {
            sk_assert(3 == callm_arg4c);
            VG_(ccall_RR_0)(cb, (Addr)check_div4,
                            callm_shadow_arg4v[1], // arg2.vseg (reg)
                            callm_shadow_arg4v[0], // arg1.vseg (reg)
                            2);

         } else if (V(idiv_64_32)) {
            sk_assert(3 == callm_arg4c);
            VG_(ccall_RR_0)(cb, (Addr)check_idiv4,
                            callm_shadow_arg4v[1], // arg2.vseg (reg)
                            callm_shadow_arg4v[0], // arg1.vseg (reg)
                            2);

         } else if (V(shldl) || V(shrdl)) {
            sk_assert(2 == callm_arg4c);
            // a) Do shrdl/shldl

         } else if (V(bsf) || V(bsr)) {
            // Shouldn't take a pointer?  Don't bother checking.
            sk_assert(2 == callm_arg4c);

         } else if (V(get_dirflag)) {
            // This always has zero pushed as the arg.
            sk_assert(1 == callm_arg4c);

         } else if (V(RDTSC))    { sk_assert(2 == callm_arg4c);
         } else if (V(CPUID))    { sk_assert(4 == callm_arg4c);
         } else if (V(SAHF))     { sk_assert(1 == callm_arg4c);
         } else if (V(LAHF))     { sk_assert(1 == callm_arg4c);
         } else if (V(fstsw_AX)) { sk_assert(1 == callm_arg4c);

         } else {
            // Others don't take any word-sized args
            //sk_assert(0 == callm_arg4c);
            if (0 != callm_arg4c) {
               VG_(printf)("callm_arg4c = %d, %d, %d\n", 
                           callm_arg4c, VGOFF_(helper_RDTSC), u->val1);
               sk_assert(1 == 0);
            }
         }

         #undef V

         VG_(copy_UInstr)(cb, u);
         break;
      //--- End of CALLM and related -----------------------------------

      case FPU:
      case MMX1: case MMX2: case MMX3:
      case SSE3: case SSE4: case SSE5:
         // a) Do FPU/MMX[123]/SSE[345]
         VG_(copy_UInstr)(cb, u);
         break;

      // We check the load, but don't do any updating, because the read is
      // into FPU/MMX/SSE regs which we don't shadow.
      case FPU_R:
      case MMX2_MemRd:
         switch (u->size) {
         case 2:   helper = (Addr)check_fpu_r2;   break;
         case 4:   helper = (Addr)check_fpu_r4;   break;
         case 8:   helper = (Addr)check_fpu_r8;   break;
         case 10:  helper = (Addr)check_fpu_r10;  break;
         case 16:  helper = (Addr)check_fpu_r16;  break;
         case 28:  helper = (Addr)check_fpu_r28;  break;
         case 108: helper = (Addr)check_fpu_r108; break;
         default:  VG_(skin_panic)("bad FPU_R/MMX/whatever size");
         }
         // FPU_R, MMX2_MemRd
         // a) Check segments match (in helper)
         // b) Do FPU_R/MMX2_MemRd
         VG_(ccall_RR_0)(cb, helper,
                         u->val2,            // m              (reg)
                         SHADOW(u->val2),    // m_ptr.vseg     (reg)
                         2);
         VG_(copy_UInstr)(cb, u);
         break;

      case FPU_W: {
      case MMX2_MemWr:
         switch (u->size) {
         case 2:   helper = (Addr)check_fpu_w2;   break;
         case 4:   helper = (Addr)check_fpu_w4;   break;
         case 8:   helper = (Addr)check_fpu_w8;   break;
         case 10:  helper = (Addr)check_fpu_w10;  break;
         case 16:  helper = (Addr)check_fpu_w16;  break;
         case 28:  helper = (Addr)check_fpu_w28;  break;
         case 108: helper = (Addr)check_fpu_w108; break;
         default:  VG_(skin_panic)("bad FPU_W/MMX/whatever size");
         }
         // FPU_W, MMX2_MemWr
         // a) Check segments match (in helper)
         // b) m.vseg = NONPTR, for each touched memory word
         // c) Do FPU_W/MMX2_MemWr
         VG_(ccall_RR_0)(cb, helper,
                         u->val2,            // m              (reg)
                         SHADOW(u->val2),    // m_ptr.vseg     (reg)
                         2);
         VG_(copy_UInstr)(cb, u);
         break;
      }

      default:
         VG_(pp_UInstr)(0, u);
         VG_(skin_panic)("Redux: unhandled instruction");
      }
      //-- End main switch ---------------------------------------------
   }

   VG_(free_UCodeBlock)(cb_in);

   // Optimisations
//   {
//      Int* live_range_ends = VG_(find_live_range_ends)(cb);
//      VG_(redundant_move_elimination)(cb, live_range_ends);
//      VG_(free_live_range_ends)(live_range_ends);
//   }

   return cb;
}


/*--------------------------------------------------------------------*/
/*--- Initialisation                                               ---*/
/*--------------------------------------------------------------------*/

void SK_(pre_clo_init)()
{
   Int i;
   // 0-terminated arrays
   Addr compact_helpers[] = {
      (Addr) do_lea1_unknown, (Addr) do_sub4,
      (Addr) check_load4,     (Addr) check_store4,
      (Addr) do_add4,         (Addr) do_and4,
      (Addr) do_or4,          (Addr) nonptr_or_unknown,
      0
   };
   Addr noncompact_helpers[] = {
      (Addr) do_lea2_1,       (Addr) do_lea2_n,
      (Addr) do_adc4,         (Addr) do_sbb4,
      (Addr) do_neg4,         (Addr) do_not4,
      (Addr) print_BB_entry,
      (Addr) check_load1,     (Addr) check_store1,
      (Addr) check_load2,     (Addr) check_store2,
      (Addr) check_imul4,     (Addr) check_mul4,
      (Addr) check_idiv4,     (Addr) check_div4,
      (Addr) check_fpu_r2,    (Addr) check_fpu_w2,
      (Addr) check_fpu_r4,    (Addr) check_fpu_w4,
      (Addr) check_fpu_r8,    (Addr) check_fpu_w8,
      (Addr) check_fpu_r10,   (Addr) check_fpu_w10,
      (Addr) check_fpu_r16,   (Addr) check_fpu_w16,
      (Addr) check_fpu_r28,   (Addr) check_fpu_w28,
      (Addr) check_fpu_r108,  (Addr) check_fpu_w108,
      0
   };

   VG_(details_name)            ("Annelid");
   VG_(details_version)         ("0.0.2");
   VG_(details_description)     ("a pointer-use checker");
   VG_(details_copyright_author)(
      "Copyright (C) 2003, and GNU GPL'd, by Nicholas Nethercote.");
   VG_(details_bug_reports_to)  ("njn25@cam.ac.uk");

   // No needs
   VG_(needs_core_errors)         ();
   VG_(needs_skin_errors)         ();
   VG_(needs_shadow_regs)         ();
   VG_(needs_command_line_options)();
   VG_(needs_syscall_wrapper)     ();
   VG_(needs_sanity_checks)       ();

   // Memory events to track
   VG_(track_new_mem_startup)      ( new_mem_startup );
   VG_(track_new_mem_stack_signal) ( NULL );
   VG_(track_new_mem_brk)          ( new_mem_brk  );
   VG_(track_new_mem_mmap)         ( new_mem_mmap );

   VG_(track_copy_mem_remap)       ( copy_mem_remap );
   VG_(track_change_mem_mprotect)  ( NULL );

   VG_(track_die_mem_stack_signal) ( NULL );
   VG_(track_die_mem_brk)          ( die_mem_brk );
   VG_(track_die_mem_munmap)       ( die_mem_munmap );

   VG_(track_pre_mem_read)         ( pre_mem_access );
   VG_(track_pre_mem_read_asciiz)  ( pre_mem_read_asciiz );
   VG_(track_pre_mem_write)        ( pre_mem_access );
   VG_(track_post_mem_write)       ( post_mem_write );

   // Register events to track
   VG_(track_post_regs_write_init)             ( post_regs_write_init      );
   VG_(track_post_reg_write_syscall_return)    ( post_reg_write_nonptr     );
   VG_(track_post_reg_write_deliver_signal)    ( post_reg_write_nonptr_or_unknown );
   VG_(track_post_reg_write_pthread_return)    ( post_reg_write_nonptr_or_unknown );
   VG_(track_post_reg_write_clientreq_return)  ( post_reg_write_nonptr     );
   VG_(track_post_reg_write_clientcall_return) ( post_reg_write_clientcall );

   // Helpers
   for (i = 0; compact_helpers[i] != 0; i++)
      VG_(register_compact_helper)( compact_helpers[i] );

   for (i = 0; noncompact_helpers[i] != 0; i++)
      VG_(register_noncompact_helper)( noncompact_helpers[i] );

   // Profiling events
   #define P(a,b) VGP_(register_profile_event)(a,b);
   P(VgpGetMemAseg,     "get-mem-aseg");
   P(VgpCheckLoadStore, "check-load-store");
   P(VgpCheckLoad4,     "check-load4");
   P(VgpCheckLoad21,    "check-load21");
   P(VgpCheckStore4,    "check-store4");
   P(VgpCheckStore21,   "check-store21");
   P(VgpCheckFpuR,      "check-fpu-r");
   P(VgpCheckFpuW,      "check-fpu-w");
   P(VgpDoAdd,          "do-add");
   P(VgpDoSub,          "do-sub");
   P(VgpDoAdcSbb,       "do-adc-sbb");
   P(VgpDoXor,          "do-xor");
   P(VgpDoAnd,          "do-and");
   P(VgpDoOr,           "do-or");
   P(VgpDoLea1,         "do-lea1");
   P(VgpDoLea2,         "do-lea2");
   #undef P

   // Other initialisation
   init_shadow_memory();
   seglist  = ISList__construct();
}

void SK_(post_clo_init)(void)
{
}

/*--------------------------------------------------------------------*/
/*--- Finalisation                                                 ---*/
/*--------------------------------------------------------------------*/

void SK_(fini)(exitcode)
{
//   if (0)
//      count_segs();
}

/*--------------------------------------------------------------------*/
/*--- end                                                an_main.c ---*/
/*--------------------------------------------------------------------*/
