// Locale support -*- C++ -*-

// Copyright (C) 1997-1999 Cygnus Solutions
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING.  If not, write to the Free
// Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.

// As a special exception, you may use this file as part of a free software
// library without restriction.  Specifically, if other files instantiate
// templates or use macros or inline functions from this file, or you compile
// this file and link it with other files to produce an executable, this
// file does not by itself cause the resulting executable to be covered by
// the GNU General Public License.  This exception does not however
// invalidate any other reasons why the executable file might be covered by
// the GNU General Public License.

//
// ISO C++ 14882: 22.1  Locales
//

#ifndef _CPP_BITS_LOCCORE_H
#define _CPP_BITS_LOCCORE_H	1

namespace std
{

  // locale header Synopsis
  /////////////////////////

  class locale;

  template<typename _Facet>
    inline const _Facet&  use_facet (const locale&);
  template<typename _Facet>
    inline bool           has_facet (const locale&) throw ();


  // _lib.locale.convenience_, convenience interfaces:
  template <typename _CharT> inline bool isspace (_CharT, const locale&);
  template <typename _CharT> inline bool isprint (_CharT, const locale&);
  template <typename _CharT> inline bool iscntrl (_CharT, const locale&);
  template <typename _CharT> inline bool isupper (_CharT, const locale&);
  template <typename _CharT> inline bool islower (_CharT, const locale&);
  template <typename _CharT> inline bool isalpha (_CharT, const locale&);
  template <typename _CharT> inline bool isdigit (_CharT, const locale&);
  template <typename _CharT> inline bool ispunct (_CharT, const locale&);
  template <typename _CharT> inline bool isxdigit(_CharT, const locale&);
  template <typename _CharT> inline bool isalnum (_CharT, const locale&);
  template <typename _CharT> inline bool isgraph (_CharT, const locale&);
  template <typename _CharT> inline _CharT toupper (_CharT, const locale&);
  template <typename _CharT> inline _CharT tolower (_CharT, const locale&);

  // _lib.category.ctype_ and _lib.facet.ctype.special_, ctype:
  class ctype_base;
  template <typename _CharT> class ctype;
  template <>             class ctype<char>;
  template <>             class ctype<wchar_t>;
  template <typename _CharT> class ctype_byname;

  class codecvt_base;
  template <typename _InternT, typename _ExternT, typename _StateT>
    class codecvt;
  template <> class codecvt<char,char,mbstate_t>;
  template <> class codecvt<wchar_t,char,mbstate_t>;
  template <typename _InternT, typename _ExternT, typename _StateT>
    class codecvt_byname;
  template <> class codecvt_byname<char,char,mbstate_t>;
  template <> class codecvt_byname<wchar_t,char,mbstate_t>;

  // _lib.category.numeric_ and _lib.facet.numpunct_, numeric:
  template <typename _Ch, typename _InIter = istreambuf_iterator<_Ch> >
    class num_get;
  template <typename _Ch, typename _OutIter = ostreambuf_iterator<_Ch> >
    class num_put;
  template <typename _CharT> class numpunct;
  template <typename _CharT> class numpunct_byname;

  // _lib.category.collate_, collation:
  template <typename _CharT> class collate;
  template <> class collate<char>;
  template <> class collate<wchar_t>;
  template <typename _CharT> class collate_byname;

  // _lib.category.time_, date and time:
  class time_base;
  template <typename _Ch, typename _InIter =  istreambuf_iterator<_Ch> >
    class time_get;
  template <typename _Ch, typename _InIter =  istreambuf_iterator<_Ch> >
    class time_get_byname;
  template <typename _Ch, typename _OutIter = ostreambuf_iterator<_Ch> >
    class time_put;
  template <typename _Ch, typename _OutIter = ostreambuf_iterator<_Ch> >
    class time_put_byname;

  // _lib.category.monetary_, money:
  class money_base;
  template <typename _CharT, typename _InIter =  istreambuf_iterator<_CharT> >
    class money_get;
  template <typename _CharT, typename _OutIter = ostreambuf_iterator<_CharT> >
    class money_put;
  template <typename _CharT, bool _Intl = false> class moneypunct;
  template <typename _CharT, bool _Intl = false> class moneypunct_byname;

  // _lib.category.messages_, message retrieval:
  class messages_base;
  template <typename _CharT> class messages;
  template <typename _CharT> class messages_byname;

  ///////////////////////////////////////////////////////////////////

  class locale
  {
  public:
    // types:
    class facet;
    class id;
    typedef int category;

    // category values
    static const category none =     0;

      // warning: much depends on the order in which these appear:
    static const category collate  = 0x0100;
    static const category ctype =    0x0200;
    static const category monetary = 0x0400;
    static const category numeric =  0x0800;
    static const category time =     0x1000;
    static const category messages = 0x2000;

    static const category all = (collate | ctype | monetary |
				 numeric | time  | messages);

    // construct/copy/destroy:
    inline  locale () throw ();
    inline  locale (const locale& __other) throw ();

    explicit  locale (const char* __std_name);
    locale (const locale& __other, const char* __std_name, category __cats);
    locale (const locale& __other, const locale& __one, category __cats);
    template<typename _Facet>
      locale (const locale& __other, _Facet* __f);
    const locale&  operator= (const locale& __other) throw ();
    inline  ~locale () throw ();  // non-virtual

    // locale operations:
    template<typename _Facet>
      locale  combine (const locale& __other);
    basic_string<char>  name () const;
    bool         operator== (const locale& __other) const throw ();
    inline bool  operator!= (const locale& __other) const throw ();
    template<typename _Ch, typename _Traits, typename _Alloc>
      bool  operator() (const basic_string<_Ch,_Traits,_Alloc>& __s1,
			const basic_string<_Ch,_Traits,_Alloc>& __s2) const;
    // global locale objects:
    static locale global (const locale&);
    static const locale& classic ();

  private:
    class _Impl;
    _Impl* _M_impl;  // the (shared) implementation

    static _Impl* _S_classic; // The one true C reference locale
    static _Impl* _S_global;  // Current global reference locale

    explicit locale (_Impl*) throw ();
    static inline void  _S_initialize ();
    static int  _S_normalize_category (int);

    static const int _S_num_categories = _Count_ones<all>::_S_count;

  public:

    friend class _Impl;
    template<typename _Facet>
      friend inline const _Facet&  use_facet (const locale&);
    template<typename _Facet>
      friend inline bool           has_facet (const locale&) throw ();
  };


  // 22.1.1.1.2  Class locale::facet

  class locale::facet
  {
  protected:
    explicit facet (size_t __refs = 0) throw ();
    virtual ~facet () {};

  private:
    size_t _M_num_references;

    void _M_add_reference () throw ();
    void _M_remove_reference () throw ();

    facet (const facet&);           // not defined
    void operator= (const facet&);  // not defined

    friend class locale;
    friend class locale::_Impl;
  };

  // 22.1.1.1.3 Class locale::id

  class locale::id
  {
  public:
    // inline id () throw ();        // does nothing (this is important)

  private:
    // note: there is no accessor for _M_index because it may
    //   be used before the constructor is run; the effect of calling a
    //   member function (even an inline) would be undefined.
    mutable size_t _M_index;
    static size_t _S_highwater;   // last id number assigned

    // void operator= (const id&);   // not defined
    // id (const id&);               // not defined

    friend class locale;
    friend class locale::_Impl;
    template<typename _Facet>
      friend inline const _Facet&  use_facet (const locale&);
    template<typename _Facet>
      friend inline bool           has_facet (const locale&) throw ();
  };

  // locale implementation object

  class locale::_Impl
  {
    size_t _M_num_references;
    vector<facet*> _M_facets;
    vector<string> _M_category_names;
    bool _M_has_name;
    bool _M_cached_name_ok;
    string _M_cached_name;

    inline void _M_add_reference () throw ();
    inline void _M_remove_reference () throw ();

    _Impl (const _Impl&, size_t __refs);
    _Impl (const _Impl&, const string&, category, size_t __refs);
    _Impl (size_t __facets, size_t __refs);
   ~_Impl () throw ();

    void _M_replace_categories (const _Impl*, category);
    void _M_replace_category (const _Impl*, const locale::id* const*);
    void _M_replace_facet (const _Impl*, const locale::id*);
    void _M_install_facet (const locale::id*, facet*);
    template<typename _Facet>
      inline void _M_init_facet (_Facet*);

    void _M_construct_collate  (const char*);
    void _M_construct_ctype    (const char*);
    void _M_construct_monetary (const char*);
    void _M_construct_numeric  (const char*);
    void _M_construct_time     (const char*);
    void _M_construct_messages (const char*);

    category _M_normalize_category_names (const string&, category __cats);

    static const locale::id* const _S_id_collate[];
    static const locale::id* const _S_id_ctype[];
    static const locale::id* const _S_id_monetary[];
    static const locale::id* const _S_id_numeric[];
    static const locale::id* const _S_id_time[];
    static const locale::id* const _S_id_messages[];

    static const locale::id* const* const _S_facet_categories[];

    friend class locale;
    friend class facet;
    template<typename _Facet>
      friend inline const _Facet&  use_facet (const locale&);
    template<typename _Facet>
      friend inline bool           has_facet (const locale&) throw ();
  };

  // locale::_Impl inlines

  inline void  locale::_Impl::_M_add_reference () throw ()
    { ++_M_num_references; }                       // XXX MT

  inline void  locale::_Impl::_M_remove_reference () throw ()
  {
    if (_M_num_references-- == 0)                   // XXX MT
      try { delete this; } catch (...) { }
  }

  template<typename _Facet>
    inline void
    locale::_Impl::_M_init_facet (_Facet* __facet)
      { _M_install_facet (&_Facet::id, __facet);  }


  // locale inlines

  inline void locale::_S_initialize ()
    { if (!_S_classic) classic ();  }

  inline locale::locale () throw ()
    { _S_initialize (); (_M_impl = _S_global)->_M_add_reference (); } // XXX MT

  inline locale::locale (const locale& __other) throw ()
    { (_M_impl = __other._M_impl)->_M_add_reference (); }

  inline locale::~locale () throw ()
    { _M_impl->_M_remove_reference (); }

  inline bool
  locale::operator!= (const locale& __other) const throw ()
    { return !(operator== (__other));  }

  struct _Bad_use_facet : public bad_cast {
     _Bad_use_facet() throw() {}
     _Bad_use_facet(_Bad_use_facet const&) throw() {}
     _Bad_use_facet& operator=(_Bad_use_facet const& __b) throw() 
	{ static_cast<bad_cast*>(this)->operator=(__b); return *this; }
     virtual char const* what() const throw();
     virtual ~_Bad_use_facet() throw();
  };

  // XXX This function is to be specialized for the "required" facets to 
  // be constructed lazily.   The specializations must be declared after 
  // the definitions of the facets themselves; but they shouldn't be 
  // inline.  Corresponding new's in locale::classic() should then be 
  // eliminated.  Note that ctype<> should not get this treatment; 
  // see the use_facet<> specializations below.
  //
  template <typename _Facet>
    const _Facet& _Use_facet_failure_handler(const locale&)
      { throw _Bad_use_facet(); }

  template<typename _Facet>
    inline const _Facet&
    use_facet (const locale& __loc)
  {
    const locale::facet* __fp = (const _Facet*)0;    // check derivation
    locale::id& __id = _Facet::id;         // check member id
    if (__id._M_index >= __loc._M_impl->_M_facets.size () ||
	(__fp = __loc._M_impl->_M_facets[__id._M_index]) == 0)
      return _Use_facet_failure_handler<_Facet>(__loc);
    return static_cast<const _Facet&>(*__fp);
  }

  // The following specializations are shortcuts: we know these facets 
  // are always present, so we can skip the tests in the generic version.

  template<>
    inline const ctype<char>&
    use_facet< const ctype<char> > (const locale& __loc);

  template<>
    inline const ctype<wchar_t>&
    use_facet< const ctype<wchar_t> > (const locale& __loc);


  template<typename _Facet>
    inline bool
    has_facet (const locale& __loc) throw ()
  {
    const locale::facet* __fp = (const _Facet*)0;   // check derivation
    locale::id& __id = _Facet::id;         // check member id
    return (__id._M_index < __loc._M_impl->_M_facets.size () &&
	    __loc._M_impl->_M_facets[__id._M_index] != 0);
  }

} // namespace std

#endif	/* _CPP_BITS_LOCCORE_H */

// Local Variables:
// mode:c++
// End:
