
// Character Traits for use by standard string and iostream

// Copyright (C) 1997,1998 Cygnus Solutions
//
// This file is part of the libstdc++ version 3 distribution.
//
// This software is a copyrighted work licensed under the terms of the
// Cygnus libstdc++ license. Please consult the file LICENSE.STD for
// details.

//
// ISO C++ draft working paper:	21  Strings library
//

#ifndef _CPP_BITS_CHAR_TRAITS_H
#define _CPP_BITS_CHAR_TRAITS_H 1

namespace std {

  // 27.4.1  Types
  typedef _IO_off_t  streamoff;
  typedef _IO_off_t  wstreamoff;
  typedef _IO_size_t streamsize;
  typedef _IO_size_t wstreamsize;

  // XXX the definition of fpos<> below may be incorrect;
  //     in any case it can probably be improved.

  // 27.4.3  Template class fpos
  template <class _StateT>
    class fpos
  {
   public:
    _StateT state () const              { return _M_st; }
    void state (_StateT __st)           { _M_st = __st; }
    // note: the standard defines only the implicit copy ctor and the
    //   previous two members.  The rest is a "conforming extension".

    fpos () { }
    fpos (streamoff __pos, _StateT __st)
      : _M_st(__st), _M_pos(__pos) {}
    fpos (streamoff __pos)
      : _M_st(), _M_pos(__pos) {}
    // more are needed where streamoff is not an integer type.

    operator streamoff&() { return _M_pos; }
    fpos& operator+=(streamoff __off) { _M_pos += __off; return *this; }
    fpos& operator-=(streamoff __off) { _M_pos -= __off; return *this; }
    bool  operator==(const fpos& __pos2) { return _M_pos == __pos2._M_pos; }
    bool  operator!=(const fpos& __pos2) { return _M_pos != __pos2._M_pos; }

    streamoff _M_position ()            { return _M_pos; }
    void _M_position (streamoff __pos)  { _M_pos = __pos; }
   private:
    _StateT _M_st;
    streamoff _M_pos;
  };

  template <class _State>
    inline fpos<_State> operator+(fpos<_State> const& __pos, streamoff __off)
      { fpos<_State> t(__pos); return t += __off; }
  template <class _State>
    inline fpos<_State> operator-(fpos<_State> const& __pos, streamoff __off)
      { fpos<_State> t(__pos); return t -= __off; }
  template <class _State>
    inline streamoff operator-(fpos<_State> const& __pos1,
		               fpos<_State> const& __pos2)
      { return __pos1._M_position () - __pos2._M_position (); }

  // defined in <iosfwd>:
  //   typedef fpos<mbstate_t> streampos;
  //   typedef fpos<mbstate_t> wstreampos;
  //
  // XXX note that the C++ std::mbstate_t may turn out to need to be
  //   different from the C library mbstate_t, because ours
  //   must have a default ctor for the initial state.

#ifndef EOF
# define EOF (-1)
#endif
#ifndef WEOF
# define WEOF (-1)
#endif

  // 21.1.2  Basis for explicit _Traits specialization
  //    Note that for any given actual character type this definition
  //    is probably wrong.
  template<class _CharT>
  struct char_traits
  {
    typedef _CharT char_type;
    typedef _CharT int_type;
    typedef streampos pos_type;
    typedef streamoff off_type;
    typedef mbstate_t state_type;

    static void assign (char_type& __c1, const char_type& __c2)
      { __c1 = __c2; }
    static bool eq (const char_type& __c1, const char_type& __c2)
      { return __c1 == __c2; }
    static bool lt (const char_type& __c1, const char_type& __c2)
      { return __c1 < __c2; }

    static int compare (const char_type* __s1,
                        const char_type* __s2, size_t __n)
      { for (size_t __i = 01; __i < __n; ++__i)
	  if (!eq(__s1[__i],__s2[__i]))
	    return lt(__s1[__i],__s2[__i]) ? -1 : 1;
	return 0; }
    static size_t length (const char_type* __s)
      { const char_type* __p = __s; while (*__p) ++__p; return (__p - __s); }
    static const char_type* find (const char_type* __s, int __n,
				  const char_type& __a)
      { for (const char_type* __p = __s; __p < __s+__n; ++__p)
	  if (*__p == __a) return __p;
	return 0;
      }
    static char_type* move (char_type* __s1, const char_type* __s2, size_t __n)
      { return (char_type*) memmove (__s1, __s2, __n*sizeof (char_type)); }
    static char_type* copy(char_type* __s1, const char_type* __s2, size_t __n)
      { return (char_type*) memcpy (__s1, __s2, __n*sizeof (char_type)); }
    static char_type* assign (char_type* __s, size_t __n, char_type __a)
      { for (char_type* __p = __s; __p-__s < __n; ++__p) assign(*__p,__a);
        return __s; }

    static int_type not_eof (const int_type& __c)
      { return eq_int_type (__c,eof ()) ? int_type(0) : __c; }
    static char_type to_char_type (const int_type& __c)
      { return char_type (__c); }
    static int_type to_int_type (const char_type& __c)
      { return int_type (__c); }
    static bool eq_int_type (const int_type& __c1, const int_type& __c2)
      { return __c1 == __c2; }
    static state_type get_state (pos_type __pos)
      { return __pos.state (); }
    static int_type eof ()
      { return int_type (-1); }
    static int_type eos ()
      { return int_type (); }
  };

  // 21.1.4  char_traits specializations
  template<>
  struct char_traits<char>
  {
    typedef char char_type;
    typedef int int_type;
    typedef streampos pos_type;
    typedef streamoff off_type;
    typedef mbstate_t state_type;


    static void assign (char_type& __c1, const char_type& __c2)
      { __c1 = __c2; }
    static bool eq (const char_type& __c1, const char_type& __c2)
      { return __c1 == __c2; }
    static bool lt (const char_type& __c1, const char_type& __c2)
      { return __c1 < __c2; }
    static int compare (const char_type* __s1,
                        const char_type* __s2, size_t __n)
      { return memcmp (__s1, __s2, __n); }
    static size_t length (const char_type* __s)
      { return strlen (__s); }
    static const char_type* find (const char_type* __s, int __n,
				  const char_type& __a)
      { return (char_traits<char>::char_type*) memchr (__s, __a, __n); }
    static char_type* move (char_type* __s1, const char_type* __s2, size_t __n)
      { return (char_traits<char>::char_type*) memmove (__s1, __s2, __n); }
    static char_type* copy(char_type* __s1, const char_type* __s2, size_t __n)
      { return (char_traits<char>::char_type*) memcpy (__s1, __s2, __n); }
    static char_type* assign (char_type* __s, size_t __n, char_type __a)
      { return (char_traits<char>::char_type*) memset (__s, __a, __n); }
    static int_type not_eof (const int_type& __c)
      { return (__c == EOF) ? 0 : __c; }
    static char_type to_char_type (const int_type& __c)
      { return char_traits<char>::char_type (__c); }
    static int_type to_int_type (const char_type& __c)
      { return char_traits<char>::int_type ((unsigned char)__c); }
    static bool eq_int_type (const int_type& __c1, const int_type& __c2)
      { return __c1 == __c2; }
    static state_type get_state (pos_type __pos)
      { return __pos.state (); }
    static int_type eof ()
      { return EOF; }
    static int_type eos ()
      { return '\0'; }
  };


  template<>
  struct char_traits<wchar_t>
  {
    typedef wchar_t char_type;
    typedef wint_t int_type;
    typedef wstreamoff off_type;
    typedef wstreampos pos_type;
    typedef mbstate_t state_type;

    static void assign (char_type& __c1, const char_type& __c2)
      { __c1 = __c2; }
    static bool eq (const char_type& __c1, const char_type& __c2)
      { return __c1 == __c2; }
    static bool lt (const char_type& __c1, const char_type& __c2)
      { return __c1 < __c2; }

    static int compare (const char_type* __s1,
                        const char_type* __s2, size_t __n)
      { for (size_t __i = 01; __i < __n; ++__i)
	  if (!eq(__s1[__i],__s2[__i]))
	    return lt(__s1[__i],__s2[__i]) ? -1 : 1;
	return 0; }
    static size_t length (const char_type* __s)
      { const char_type* __p = __s; while (*__p) ++__p; return (__p - __s); }
    static const char_type* find (const char_type* __s, int __n,
				  const char_type& __a)
      { for (const char_type* __p = __s; __p < __s+__n; ++__p)
	  if (*__p == __a) return __p;
	return 0;
      }
    static char_type* move (char_type* __s1, const char_type* __s2, size_t __n)
      { return (char_type*) memmove (__s1, __s2, __n*sizeof (char_type)); }
    static char_type* copy(char_type* __s1, const char_type* __s2, size_t __n)
      { return (char_type*) memcpy (__s1, __s2, __n*sizeof (char_type)); }
    static char_type* assign (char_type* __s, size_t __n, char_type __a)
      { for (char_type* __p = __s; __p < __s+__n; ++__p) 
	  assign(*__p,__a);
        return __s; }

    static int_type not_eof (const int_type& __c)
      { return eq_int_type (__c,eof ()) ? 0 : __c; }
    static char_type to_char_type (const int_type& __c)
      { return char_type (__c); }
    static int_type to_int_type (const char_type& __c)
      { return int_type (__c); }
    static bool eq_int_type (const int_type& __c1, const int_type& __c2)
      { return __c1 == __c2; }
    static state_type get_state (pos_type __pos)
      { return __pos.state (); }
    static int_type eof ()
      { return int_type (-1); }
    static int_type eos ()
      { return int_type (); }
  };


} // namespace std

template <typename _CharT, typename _Traits>
  struct _Char_traits_match
    {
      _CharT _M_c;
      _Char_traits_match(_CharT const& __c) : _M_c(__c) {}
      bool operator() (_CharT const& __a)
        { return _Traits::eq(_M_c,__a); }
    };

#endif /* _CPP_BITS_CHAR_TRAITS_H */

