// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1989 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

/* *** Version 1.2 -- nearly 100% AT&T 1.2 compatible *** */

#ifndef _stream_h
#pragma once
#define _stream_h 1

/* uncomment the next line to disable    ostream << char */
//#define NO_OUTPUT_CHAR


#include <File.h>
#include <streambuf.h>

class whitespace                // a class used only to input and
{                               // discard white space characters
  char filler;                     
};

class istream;

class ostream
{
  friend class istream;
protected:
  streambuf*    bp;
  state_value   state;           // _good/_eof/_fail/_bad
  char          ownbuf;          // true if we own *bp
  
public:
                ostream(const char* filename, io_mode m, access_mode a);
                ostream(const char* filename, const char* m);
                ostream(int filedesc, io_mode m);
                ostream(FILE* fileptr);
                ostream(int sz, char* buf);
                ostream(int filedesc, char* buf, int buflen);
                ostream(int filedesc);
                ostream(streambuf* s);

               ~ostream();

  ostream&      open(const char* filename, io_mode m, access_mode a);
  ostream&      open(const char* filename, const char* m);
  ostream&      open(int  filedesc, io_mode m);
  ostream&      open(FILE* fileptr);
  ostream&      open(const char* filenam, open_mode m);

  ostream&      close();
  ostream&      flush();

// stream status

  int           rdstate();
  int           eof();
  int           fail();
  int           bad();
  int           good();

// other status queries

  int           readable();
  int           writable();
  int           is_open();

  void*         operator void*();
  int           operator !();

  const char*   name();

  char*         bufptr();

// error handling

  void          error();
  void          clear(state_value f = 0); // poorly named
  ostream&      failif(int cond);

// unformatted IO

  ostream&      put(char  c);
  ostream&      put(const char* s);
  ostream&      put(const char* s, int slen);
           
// formatted IO

  ostream&      form(const char* fmt, ...);           

  ostream&      operator << (short  n);
  ostream&      operator << (unsigned short n);
  ostream&      operator << (int    n);
  ostream&      operator << (unsigned int n);
  ostream&      operator << (long   n);
  ostream&      operator << (unsigned long n);
  ostream&      operator << (float  n);
  ostream&      operator << (double n);
  ostream&      operator << (const char* s);

#ifndef NO_OUTPUT_CHAR
  ostream&      operator << (char   c);
#endif

};


class istream
{
  friend void   eatwhite(istream& s);
protected:
  streambuf*    bp;
  state_value   state;           // _good/_eof/_fail/_bad
  ostream*      tied_to;
  char          skipws;
  char          ownbuf;
  void          _flush();
  char*         readline (int chunk_number, char terminator);
  
public:
                istream(const char* filename, io_mode m, access_mode a, 
                        int sk=1, ostream* t = 0);
                istream(const char* filename, const char* m, 
                        int sk=1, ostream* t = 0);
                istream(int filedesc, io_mode m, int sk=1, ostream* t = 0);
                istream(FILE* fileptr, int sk=1, ostream* t = 0);
                istream(int sz, char* buf, int sk=1, ostream* t = 0);
                istream(int filedesc, int sk=1, ostream* t = 0);
                istream(int filedesc, char* buf, int buflen, 
                        int sk=1, ostream* t = 0);
                istream(streambuf* s, int sk=1, ostream* t = 0);

               ~istream();

  istream&      open(const char* filename, io_mode m, access_mode a);
  istream&      open(const char* filename, const char* m);
  istream&      open(int  filedesc, io_mode m);
  istream&      open(FILE* fileptr);
  istream&      open(const char* filenam, open_mode m);

  istream&      close();

  ostream*      tie(ostream* s);
  int           skip(int);

// stream status

  int           rdstate();
  int           eof();
  int           fail();
  int           bad();
  int           good();

// other status queries

  int           readable();
  int           writable();
  int           is_open();

  void*         operator void*();
  int           operator !();

  const char*   name();

  char*         bufptr();

// error handling

  void          error();
  void          clear(state_value f = 0); // poorly named
  istream&      failif(int cond);

// unformatted IO

  istream&      get(char& c);
  istream&      unget(char c);
  istream&      putback(char c); // a synonym for unget

  istream&      get    (char* s, int n, char terminator = '\n');
  istream&      getline(char* s, int n, char terminator = '\n');
  istream&      gets   (char **s, char terminator = '\n');


  istream&      operator >> (char&   c);
  istream&      operator >> (short&  n);
  istream&      operator >> (unsigned short& n);
  istream&      operator >> (int&    n);
  istream&      operator >> (unsigned int& n);
  istream&      operator >> (long&   n);
  istream&      operator >> (unsigned long& n);
  istream&      operator >> (float&  n);
  istream&      operator >> (double& n);
  istream&      operator >> (char*   s);
  istream&      operator >> (whitespace& w);
};

// pre-declared streams

extern istream  cin;             // stdin
extern ostream  cout;            // stdout
extern ostream  cerr;            // stderr

extern whitespace WS;            // for convenience


inline void ostream::clear(state_value flag = _good)
{
  state = flag;
}

inline int ostream::rdstate()
{
  return int(state);
}

inline int ostream::good()
{
  return state == _good;
}

inline int ostream::eof()
{
  return int(state) & int(_eof);
}

inline int ostream::fail()
{
  return int(state) & int(_fail);
}

inline int ostream::bad()
{
  return int(state) & int(_bad);
}

inline void* ostream::operator void*()
{
  return (state == _good)? this : 0;
}

inline int ostream::operator !()
{
  return (state != _good);
}

inline ostream& ostream::failif(int cond)
{
  if (cond) state |= _fail; return *this;
}

inline int ostream::is_open()
{
  return bp->is_open();
}

inline int ostream::readable()
{
  return 0;
}

inline int ostream::writable()
{
  return (bp != 0) && (state == _good);
}


inline char* ostream::bufptr()
{
  return bp->base;
}

inline ostream& ostream::flush()
{
  bp->overflow(); return *this;
}

inline ostream& ostream::close()
{
  bp->overflow(); bp->close();  return *this;
}

inline ostream& ostream::put(char ch)
{
  return failif((state != _good) || bp->sputc(ch) == EOF);
}

#ifndef NO_OUTPUT_CHAR
inline ostream& ostream::operator << (char ch)
{
  return failif((state != _good) || bp->sputc(ch) == EOF);
}
#endif

inline ostream& ostream::put(const char* s)
{
  return failif((state != _good) || bp->sputs(s) == EOF);
}

inline ostream& ostream::put(const char* s, int len)
{
  return failif((state != _good) || bp->sputsn(s, len) == EOF);
}

inline ostream& ostream::operator << (const char* s)
{
  return failif((state != _good) || bp->sputs(s) == EOF);
}


  

//-----------------------------------------------------------------

inline void istream::clear(state_value flag = _good)
{
  state = flag;
}

inline int istream::rdstate()
{
  return int(state);
}

inline int istream::good()
{
  return state == _good;
}

inline int istream::eof()
{
  return int(state) & int(_eof);
}

inline int istream::fail()
{
  return int(state) & int(_fail);
}

inline int istream::bad()
{
  return int(state) & int(_bad);
}

inline void* istream::operator void*()
{
  return (state == _good)? this : 0;
}

inline int istream::operator !()
{
  return (state != _good);
}

inline istream& istream::failif(int cond)
{
  if (cond) state |= _fail; return *this;
}

inline int istream::is_open()
{
  return bp->is_open();
}

inline int istream::readable()
{
  return (bp != 0) && (bp->is_open()) && (state == _good);
}

inline int istream::writable()
{
  return 0;
}

inline void istream::_flush()
{
  if(tied_to != 0) tied_to->flush();
}


inline char* istream::bufptr()
{
  return bp->base;
}


inline istream& istream::close()
{
  bp->close();  return *this;
}


inline int istream::skip(int sk)
{
  int was = skipws; skipws = sk; return was;
}

inline ostream* istream::tie(ostream* s)
{
  ostream* was = tied_to; tied_to = s; return was;
}

inline istream& istream::unget(char c)
{
  bp->sputback(c); return *this;
}

inline istream& istream::putback(char c)
{
  bp->sputback(c); return *this;
}

inline void eatwhite(istream& s)
{
  s >> WS;
}

#endif
