// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/


#ifndef _String_h
#pragma once
#define _String_h 1

#include <stream.h>

struct StrRep                     // internal String representations
{
  unsigned short    len;         // string length 
  unsigned short    sz;          // allocated space
  char              s[1];        // the string starts here 
                                 // (at least 1 char for trailing null)
                                 // allocated & expanded via non-public fcts

  friend StrRep*     Salloc(StrRep*, const char*, int, int);
  friend StrRep*     Scopy(StrRep*, StrRep*);
  friend StrRep*     Sresize(StrRep*, int);
  friend StrRep*     Scat(StrRep*, const char*, int, const char*, int);
  friend StrRep*     Sprepend(StrRep*, const char*, int);
  friend StrRep*     Sreverse(StrRep*, StrRep*);
  friend StrRep*     Supcase(StrRep*, StrRep*);
  friend StrRep*     Sdowncase(StrRep*, StrRep*);
  friend StrRep*     Scapitalize(StrRep*, StrRep*);

};


class String;
class SubString;
class StrTmp;


struct re_pattern_buffer;       // defined elsewhere
struct re_registers;

class Regex
{
  friend class       String;
  friend class       SubString;
protected:
  re_pattern_buffer* buf;
  re_registers*      reg;

  void               initialize(const char* t, int tlen, int fast, 
                                int bufsize, const char* transtable);

public:
                     Regex(const char* t, 
                           int fast = 0, 
                           int bufsize = 40, 
                           const char* transtable = 0);

                     Regex(String& x, 
                           int fast = 0, 
                           int bufsize = 40, 
                           const char* transtable = 0);

                     ~Regex();

  int                match(const char* s, int len, int pos = 0);
  int                search(const char* s, int len, 
                            int& matchlen, int startpos = 0);
  int                match_info(int& start, int& length, int nth = 0);

  int                OK();  // representation invariant
};



class SubString
{
  friend class      String;
  friend class      StrTmp;
protected:
  String*           S;
  unsigned short    pos;
  unsigned short    len;

  void              assign(StrRep*, const char*, int = -1);
                    SubString(String* x, int p, int l);
                    SubString(const SubString& x);

public:
                    ~SubString();

  void              operator =  (String&     y);
  void              operator =  (SubString&  y);
  void              operator =  (const char* t);
  void              operator =  (char        c);
  
  StrTmp            operator +  (String&     y);
  StrTmp            operator +  (SubString&  y);
  StrTmp            operator +  (const char* t);
  StrTmp            operator +  (char        c);
  StrTmp            operator +  (StrTmp& y);
  friend StrTmp     operator +  (const char* t,SubString&  y);
  friend StrTmp     operator +  (char       c, SubString&  x);

  friend int        operator == (SubString& x, String&     y);
  friend int        operator == (String&    x, SubString&  y);
  friend int        operator == (SubString& x, SubString&  y);
  friend int        operator == (SubString& x, const char* t);
  
  friend int        operator != (SubString& x, String&     y);
  friend int        operator != (String&    x, SubString&  y);
  friend int        operator != (SubString& x, SubString&  y);
  friend int        operator != (SubString& x, const char* t);
  
  friend int        operator <= (SubString& x, String&     y);
  friend int        operator <= (String&    x, SubString&  y);
  friend int        operator <= (SubString& x, SubString&  y);
  friend int        operator <= (SubString& x, const char* t);
  
  friend int        operator <  (SubString& x, String&     y);
  friend int        operator <  (String&    x, SubString&  y);
  friend int        operator <  (SubString& x, SubString&  y);
  friend int        operator <  (SubString& x, const char* t);
  
  friend int        operator >= (SubString& x, String&     y);
  friend int        operator >= (String&    x, SubString&  y);
  friend int        operator >= (SubString& x, SubString&  y);
  friend int        operator >= (SubString& x, const char* t);
  
  friend int        operator >  (SubString& x, String&     y);
  friend int        operator >  (String&    x, SubString&  y);
  friend int        operator >  (SubString& x, SubString&  y);
  friend int        operator >  (SubString& x, const char* t);

  int               contains(char        c);
  int               contains(String&     y);
  int               contains(SubString&  y);
  int               contains(const char* t);
  int               contains(Regex&       r);

  int               matches(Regex&  r);

// misc

  int               length();
  int               empty();

  friend ostream&   operator<<(ostream& s, SubString& x);

  int               OK(); 

// non-operator versions

  friend int        compare(SubString& x, String&     y);
  friend int        compare(String&    x, SubString&  y);
  friend int        compare(SubString& x, SubString&  y);
  friend int        compare(SubString& x, const char* y);

};


class String
{
  friend class      SubString;
  friend class      Regex;
  friend class      StrTmp;

protected:
  StrRep*           rep;

  int               search(int, int, const char*, int = -1);
  int               search(int, int, char);
  int               match(int, int, int, const char*, int = -1);
  int               _gsub(const char*, int, const char* ,int);
  int               _gsub(Regex&, const char*, int);

public:

// constructors & assignment

                    String();
                    String(String& x);
                    String(SubString&  x);
                    String(const char* t);
                    String(const char* t, int len);
                    String(char c);

                    ~String();

  void              operator =  (String&     y);
  void              operator =  (StrTmp&     y);
  void              operator =  (const char* y);
  void              operator =  (char        c);
  void              operator =  (SubString&  y);

// concatenation

  StrTmp            operator +  (String&     y);     
  StrTmp            operator +  (const char* t);
  StrTmp            operator +  (char        c);
  StrTmp            operator +  (SubString&  y);     
  StrTmp            operator +  (StrTmp& y);
  friend StrTmp     operator +  (const char* t, String& y);
  friend StrTmp     operator +  (char        c, String& x);

  void              operator += (String&     y); 
  void              operator += (SubString&  y);
  void              operator += (const char* t);
  void              operator += (char        c);

  void              prepend(String&     y); 
  void              prepend(SubString&  y);
  void              prepend(const char* t);
  void              prepend(char        c);

  
// relational operators

  friend int        operator == (String&     x, String&     y);
  friend int        operator == (String&     x, const char* t);
  friend int        operator == (String&     x, SubString&  y);
  friend int        operator == (SubString&  x, String&     y);
  
  friend int        operator != (String&     x, String&     y);
  friend int        operator != (String&     x, const char* t);
  friend int        operator != (String&     x, SubString&  y);
  friend int        operator != (SubString&  x, String&     y);
  
  friend int        operator <= (String&     x, String&     y);
  friend int        operator <= (String&     x, const char* t);
  friend int        operator <= (String&     x, SubString&  y);
  friend int        operator <= (SubString&  x, String&     y);
  
  friend int        operator <  (String&     x, String&     y);
  friend int        operator <  (String&     x, const char* t);
  friend int        operator <  (String&     x, SubString&  y);
  friend int        operator <  (SubString&  x, String&     y);
  
  friend int        operator >= (String&     x, String&     y);
  friend int        operator >= (String&     x, const char* t);
  friend int        operator >= (String&     x, SubString&  y);
  friend int        operator >= (SubString&  x, String&     y);
  
  friend int        operator >  (String&     x, String&     y);
  friend int        operator >  (String&     x, const char* t);
  friend int        operator >  (String&     x, SubString&  y);
  friend int        operator >  (SubString&  x, String&     y);

  friend int        fcompare(String&   x, String&     y); // ignore case

// searching & matching

  int               index(char        c, int startpos = 0);      
  int               index(String&     y, int startpos = 0);      
  int               index(SubString&  y, int startpos = 0);      
  int               index(const char* t, int startpos = 0);  
  int               index(Regex&      r, int startpos = 0);       

  int               contains(char        c);
  int               contains(String&     y);
  int               contains(SubString&  y);
  int               contains(const char* t);
  int               contains(Regex&      r);

  int               contains(char        c, int pos);
  int               contains(String&     y, int pos);
  int               contains(SubString&  y, int pos);
  int               contains(const char* t, int pos);
  int               contains(Regex&      r, int pos);

  int               matches(char        c, int pos = 0);
  int               matches(String&     y, int pos = 0);
  int               matches(SubString&  y, int pos = 0);
  int               matches(const char* t, int pos = 0);
  int               matches(Regex&      r, int pos = 0);

  int               freq(char        c); // return number of c's in string
  int               freq(String&     y);
  int               freq(SubString&  y);
  int               freq(const char* t);

// substring extraction

  SubString         at(int         pos, int len);
  SubString         at(String&     x, int startpos = 0); 
  SubString         at(SubString&  x, int startpos = 0); 
  SubString         at(const char* t, int startpos = 0);
  SubString         at(char        c, int startpos = 0);
  SubString         at(Regex&      r, int startpos = 0); 

  SubString         before(int          pos);
  SubString         before(String&      x, int startpos = 0);
  SubString         before(SubString&   x, int startpos = 0);
  SubString         before(const char*  t, int startpos = 0);
  SubString         before(char         c, int startpos = 0);
  SubString         before(Regex&       r, int startpos = 0);

  SubString         through(int          pos);
  SubString         through(String&      x, int startpos = 0);
  SubString         through(SubString&   x, int startpos = 0);
  SubString         through(const char*  t, int startpos = 0);
  SubString         through(char         c, int startpos = 0);
  SubString         through(Regex&       r, int startpos = 0);

  SubString         from(int          pos);
  SubString         from(String&      x, int startpos = 0);
  SubString         from(SubString&   x, int startpos = 0);
  SubString         from(const char*  t, int startpos = 0);
  SubString         from(char         c, int startpos = 0);
  SubString         from(Regex&       r, int startpos = 0);

  SubString         after(int         pos);
  SubString         after(String&     x, int startpos = 0);
  SubString         after(SubString&  x, int startpos = 0);
  SubString         after(const char* t, int startpos = 0);
  SubString         after(char        c, int startpos = 0);
  SubString         after(Regex&      r, int startpos = 0);

// modification

  void              del(int         pos, int len);
  void              del(String&     y, int startpos = 0);
  void              del(SubString&  y, int startpos = 0);
  void              del(const char* t, int startpos = 0);
  void              del(char        c, int startpos = 0);
  void              del(Regex&      r, int startpos = 0);

  int               gsub(String&     pat, String&     repl);
  int               gsub(SubString&  pat, String&     repl);
  int               gsub(const char* pat, String&     repl);
  int               gsub(const char* pat, const char* repl);
  int               gsub(Regex&      pat, String&     repl);

// friends & utilities

  friend int        split(String& x, String res[], int maxn, String& sep);
  friend int        split(String& x, String res[], int maxn, Regex&  sep);

  friend StrTmp     join(String src[], int n, String& sep);

  friend StrTmp     replicate(char        c, int n);
  friend StrTmp     replicate(String&     y, int n);

  friend StrTmp     common_prefix(String& x, String& y, int startpos = 0);
  friend StrTmp     common_suffix(String& x, String& y, int startpos = -1);

  friend StrTmp     reverse(String& x);
  friend StrTmp     upcase(String& x);
  friend StrTmp     downcase(String& x);
  friend StrTmp     capitalize(String& x);

// in-place versions of above

  void              reverse();
  void              upcase();
  void              downcase();
  void              capitalize();

// conversion

  char&             operator [] (int i);

  const char*       operator char*();

// IO

  friend ostream&   operator<<(ostream& s, String& x);
  friend ostream&   operator<<(ostream& s, SubString& x);
  friend istream&   operator>>(istream& s, String& x);

  friend int        readline(istream& s, String& x, 
                             char terminator = '\n',
                             int discard_terminator = 1);

// status

  int               length();
  int               empty();

  void              alloc(int newsize);

  void              error(char* msg);
  int               OK();

// non-operator versions of operator functions

  friend int        compare(String&    x, String&     y);
  friend int        compare(String&    x, SubString&  y);
  friend int        compare(String&    x, const char* y);
  friend int        compare(SubString& x, String&     y);
};

class StrTmp : public String
{
public:
                    StrTmp(StrRep* p);
                    StrTmp(String& x);
                    StrTmp(StrTmp& x);
                    ~StrTmp();

  StrTmp            operator + (String& y); 
  StrTmp            operator + (SubString& y); 
  StrTmp            operator + (const char* y); 
  StrTmp            operator + (char y); 
  friend StrTmp     operator + (const char* x, StrTmp& y);
  friend StrTmp     operator + (char x, StrTmp& y);

  friend StrTmp     reverse(StrTmp& x);
  friend StrTmp     upcase(StrTmp& x);
  friend StrTmp     downcase(StrTmp& x);
  friend StrTmp     capitalize(StrTmp& x);

};

// some built in regular expressions

extern Regex RXwhite;          // = "[ \n\t]+"
extern Regex RXint;            // = "-?[0-9]+"
extern Regex RXdouble;         // = "-?\\(\\([0-9]+\\.[0-9]*\\)\\|
                               //    \\([0-9]+\\)\\|\\(\\.[0-9]+\\)\\)
                               //    \\([eE][---+]?[0-9]+\\)?"
extern Regex RXalpha;          // = "[A-Za-z]+"
extern Regex RXlowercase;      // = "[a-z]+"
extern Regex RXuppercase;      // = "[A-Z]+"
extern Regex RXalphanum;       // = "[0-9A-Za-z]+"
extern Regex RXidentifier;     // = "[A-Za-z_][A-Za-z0-9_]*"


//#ifdef __OPTIMIZE__


extern StrRep  _nilStrRep;
extern String _nilString;


inline String::String()
{ 
  rep = &_nilStrRep;
}

inline String::String(String& x)
{ 
  rep = Scopy(0, x.rep);
}

inline String::String(const char* t)
{
  rep = Salloc(0, t, -1, -1);
}

inline StrTmp::StrTmp(StrRep* r) 
{
  rep = r;
}

inline StrTmp::StrTmp(String& x) 
{
  rep = x.rep; x.rep = &_nilStrRep;
}

inline StrTmp::StrTmp(StrTmp& x) 
{
  rep = x.rep; x.rep = &_nilStrRep;
}

inline String::String(const char* t, int tlen)
{
  rep = Salloc(0, t, tlen, tlen);
}

inline String::String(SubString& y)
{
  rep = Salloc(0, &(y.S->rep->s[y.pos]), y.len, y.len);
}

inline String::String(char c)
{
  rep = Salloc(0, &c, 1, 1);
}

inline String::~String()
{ 
  if (rep != &_nilStrRep) delete rep;
}

inline StrTmp::~StrTmp() {} 

inline void String::operator =  (String& y)
{ 
  rep = Scopy(rep, y.rep);
}

inline void String::operator =  (StrTmp& y)
{ 
  if (rep != &_nilStrRep) delete rep; 
  rep = y.rep; y.rep = &_nilStrRep;
}

inline void String::operator=(const char* t)
{
  rep = Salloc(rep, t, -1, -1); 
}

inline void String::operator=(SubString&  y)
{
  rep = Salloc(rep, &(y.S->rep->s[y.pos]), y.len, y.len);
}

inline void String::operator=(char c)
{
  rep = Salloc(rep, &c, 1, 1); 
}

inline void String::operator +=(String& y)
{
  rep = Scat(rep, rep->s, rep->len, y.rep->s, y.rep->len);
}

inline void String::operator +=(SubString& y)
{
  rep = Scat(rep, rep->s, rep->len, &(y.S->rep->s[y.pos]),y.len);
}

inline void String::operator += (const char* y)
{
  rep = Scat(rep, rep->s, rep->len, y, -1);
}

inline void String:: operator +=(char y)
{
  rep = Scat(rep, rep->s, rep->len, &y, 1); 
}

inline StrTmp String::operator + (String& y)
{
  return(Scat(0, rep->s, rep->len, y.rep->s, y.rep->len));
}

inline StrTmp String::operator +(SubString& y)
{
  return(Scat(0, rep->s, rep->len, &(y.S->rep->s[y.pos]),y.len));
}

inline StrTmp String::operator + (const char* y)
{
  return(Scat(0, rep->s, rep->len, y, -1));
}

inline StrTmp String::operator + (char y)
{
  return(Scat(0, rep->s, rep->len, &y, 1));
}

inline StrTmp SubString::operator + (String& y)
{
  return(Scat(0, &(S->rep->s[pos]), len, y.rep->s, y.rep->len));
}

inline StrTmp SubString::operator + (SubString& y)
{
  return(Scat(0, &(S->rep->s[pos]), len, &(y.S->rep->s[y.pos]), y.len));
}

inline StrTmp SubString::operator + (const char* y)
{
  return(Scat(0, &(S->rep->s[pos]), len, y, -1));
}

inline StrTmp SubString::operator + (char y)
{
  return(Scat(0, &(S->rep->s[pos]), len, &y, 1));
}

inline StrTmp operator +(const char* t, String& y)
{
  return(Scat(0, t, -1, y.rep->s, y.rep->len));
}

inline StrTmp operator + (const char* t, SubString& y)
{
  return(Scat(0, t, -1, &(y.S->rep->s[y.pos]), y.len));
}

inline StrTmp operator + (char c, String& y)
{
  return(Scat(0, &c, 1, y.rep->s, y.rep->len));
}

inline StrTmp operator + (char c, SubString& y)
{
  return(Scat(0, &c, 1, &(y.S->rep->s[y.pos]), y.len));
}

inline StrTmp StrTmp::operator + (String& y)
{
  rep = Scat(rep, rep->s, rep->len, y.rep->s, y.rep->len); return *this;
}

inline StrTmp StrTmp::operator + (SubString& y)
{
  rep = Scat(rep, rep->s, rep->len, &(y.S->rep->s[y.pos]),y.len); return *this;
}

inline StrTmp StrTmp::operator + (const char* y)
{
  rep = Scat(rep, rep->s, rep->len, y, -1); return *this;
}

inline StrTmp StrTmp::operator + (char y)
{
  rep = Scat(rep, rep->s, rep->len, &y, 1); return *this;
}

inline StrTmp String::operator + (StrTmp& y)
{
  y.rep = Sprepend(y.rep, rep->s, rep->len); return y;
}

inline StrTmp SubString::operator + (StrTmp& y)
{
  y.rep = Sprepend(y.rep, &(S->rep->s[pos]), len); return y;
}

inline StrTmp operator + (const char* x, StrTmp& y)
{
  y.rep = Sprepend(y.rep, x, -1); return y;
}

inline StrTmp operator + (char x, StrTmp& y)
{
  y.rep = Sprepend(y.rep, &x, 1); return y;
}

inline void String::prepend(String& y)
{
  rep = Sprepend(rep, y.rep->s, y.rep->len);
}

inline void String::prepend(const char* y)
{
  rep = Sprepend(rep, y, -1); 
}

inline void String::prepend(char y)
{
  rep = Sprepend(rep, &y, 1); 
}

inline void String::prepend(SubString& y)
{
  rep = Sprepend(rep, &(y.S->rep->s[y.pos]), y.len);
}

inline StrTmp reverse(String& x)
{
  return(Sreverse(x.rep, 0));
}

inline StrTmp reverse(StrTmp& x)
{
  x.rep = Sreverse(x.rep, x.rep); return x;
}

inline void String::reverse()
{
  rep = Sreverse(rep, rep);
}

inline StrTmp upcase(String& x)
{
  return(Supcase(x.rep, 0));
}

inline StrTmp upcase(StrTmp& x)
{
  x.rep = Supcase(x.rep, x.rep); return x;
}

inline void String::upcase()
{
  rep = Supcase(rep, rep);
}

inline StrTmp downcase(String& x)
{
  return(Sdowncase(x.rep, 0));
}

inline StrTmp downcase(StrTmp& x)
{
  x.rep = Sdowncase(x.rep, x.rep); return x;
}

inline void String::downcase()
{
  rep = Sdowncase(rep, rep);
}

inline StrTmp capitalize(String& x)
{
  return(Scapitalize(x.rep, 0));
}

inline StrTmp capitalize(StrTmp& x)
{
  x.rep = Scapitalize(x.rep, x.rep); return x;
}

inline void String::capitalize()
{
  rep = Scapitalize(rep, rep);
}

inline void String::alloc(int newsize)
{
  rep = Sresize(rep, newsize);
}

inline SubString::SubString(const SubString& x)
{ 
  S = x.S; pos = x.pos;   len = x.len; 
}

inline SubString::SubString(String* x, int first, int l)
{
  if (first < 0 || (unsigned)(first + l) > x->rep->len)
  {
    S = &_nilString; pos = len = 0;
  }
  else
  {
    S = x; pos = first; len = l;
  }
}

inline SubString::~SubString() {}

inline void SubString::operator = (const char* ys)
{
  assign(0, ys);
}

inline void SubString::operator = (char ch)
{
  assign(0, &ch, 1);
}

inline void SubString::operator = (String& y)
{
  assign(y.rep, y.rep->s, y.rep->len);
}

inline void SubString::operator = (SubString& y)
{
  assign(y.S->rep, &(y.S->rep->s[y.pos]), y.len);
}

inline int String::length()
{ 
  return rep->len;
}

inline int String::empty()
{ 
  return rep->len == 0;
}

inline char&  String::operator [] (int i) 
{ 
  if (((unsigned)i) >= rep->len) error("invalid index");
  return rep->s[i];
}

inline int String::index(char c, int startpos = 0)
{
  return search(startpos, rep->len, c);
}

inline int String::index(const char* t, int startpos = 0)
{   
  return search(startpos, rep->len, t);
}

inline int String::index(String& y, int startpos = 0)
{   
  return search(startpos, rep->len, y.rep->s, y.rep->len);
}

inline int String::index(SubString& y, int startpos = 0)
{   
  return search(startpos, rep->len, &(y.S->rep->s[y.pos]), y.len);
}

inline int String::contains(char c)
{
  return search(0, rep->len, c) >= 0;
}

inline int SubString::contains(char c)
{
  return S->search(pos, pos+len, 0, c) >= 0;
}

inline int String::contains(const char* t)
{   
  return search(0, rep->len, t) >= 0;
}

inline int String::contains(String& y)
{   
  return search(0, rep->len, y.rep->s, y.rep->len) >= 0;
}

inline int String::contains(SubString& y)
{   
  return search(0, rep->len, &(y.S->rep->s[y.pos]), y.len) >= 0;
}

inline int SubString::contains(const char* t)
{   
  return S->search(pos, pos+len, t) >= 0;
}

inline int SubString::contains(String& y)
{   
  return S->search(pos, pos+len, y.rep->s, y.rep->len) >= 0;
}

inline int SubString::contains(SubString&  y)
{   
  return S->search(pos, pos+len, &(y.S->rep->s[y.pos]), y.len) >= 0;
}

inline int String::contains(char c, int p)
{
  return match(p, rep->len, 0, &c, 1);
}

inline int String::matches(char c, int p = 0)
{
  return match(p, rep->len, 1, &c, 1);
}

inline int String::contains(const char* t, int p)
{
  return match(p, rep->len, 0, t);
}

inline int String::matches(const char* t, int p = 0)
{
  return match(p, rep->len, 1, t);
}

inline int String::contains(String& y, int p)
{
  return match(p, rep->len, 0, y.rep->s, y.rep->len);
}

inline int String::matches(String& y, int p = 0)
{
  return match(p, rep->len, 1, y.rep->s, y.rep->len);
}

inline int String::contains(SubString& y, int p)
{
  return match(p, rep->len, 0, &(y.S->rep->s[y.pos]), y.len);
}

inline int String::matches(SubString& y, int p = 0)
{
  return match(p, rep->len, 1, &(y.S->rep->s[y.pos]), y.len);
}

inline int String::contains(Regex& r)
{
  int unused;  return r.search(rep->s, rep->len, unused, 0) >= 0;
}

inline int SubString::contains(Regex& r)
{
  int unused;  return r.search(&(S->rep->s[pos]), len, unused, 0) >= 0;
}

inline int String::contains(Regex& r, int p)
{
  return r.match(rep->s, rep->len, p) >= 0;
}

inline int String::matches(Regex& r, int p = 0)
{
  int l = (p < 0)? -p : rep->len - p;
  return r.match(rep->s, rep->len, p) == l;
}

inline int SubString::matches(Regex& r)
{
  return r.match(&(S->rep->s[pos]), len, 0) == len;
}

inline const char* String::operator char*()
{ 
  return rep->s;
}

inline int String::index(Regex& r, int startpos = 0)
{
  int unused;  return r.search(rep->s, rep->len, unused, startpos);
}


inline  int SubString::length()
{ 
  return len;
}

inline  int SubString::empty()
{ 
  return len == 0;
}

inline  ostream& operator<<(ostream& s, String& x)
{ 
  s.put(x.rep->s); return s;
}

inline int operator==(String& x, String& y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(String& x, String& y)
{
  return compare(x, y) != 0; 
}

inline int operator>(String& x, String& y)
{
  return compare(x, y) > 0; 
}

inline int operator>=(String& x, String& y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(String& x, String& y)
{
  return compare(x, y) < 0; 
}

inline int operator<=(String& x, String& y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(String& x, SubString&  y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(String& x, SubString&  y)
{
  return compare(x, y) != 0; 
}

inline int operator>(String& x, SubString&  y)      
{
  return compare(x, y) > 0; 
}

inline int operator>=(String& x, SubString&  y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(String& x, SubString&  y) 
{
  return compare(x, y) < 0; 
}

inline int operator<=(String& x, SubString&  y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(String& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(String& x, const char* t) 
{
  return compare(x, t) != 0; 
}

inline int operator>(String& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(String& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(String& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(String& x, const char* t) 
{
  return compare(x, t) <= 0; 
}

inline int operator==(SubString& x, String& y) 
{
  return compare(y, x) == 0; 
}

inline int operator!=(SubString& x, String& y)
{
  return compare(y, x) != 0;
}

inline int operator>(SubString& x, String& y)      
{
  return compare(y, x) < 0;
}

inline int operator>=(SubString& x, String& y)     
{
  return compare(y, x) <= 0;
}

inline int operator<(SubString& x, String& y)      
{
  return compare(y, x) > 0;
}

inline int operator<=(SubString& x, String& y)     
{
  return compare(y, x) >= 0;
}

inline int operator==(SubString& x, SubString&  y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(SubString& x, SubString&  y)
{
  return compare(x, y) != 0;
}

inline int operator>(SubString& x, SubString&  y)      
{
  return compare(x, y) > 0;
}

inline int operator>=(SubString& x, SubString&  y)
{
  return compare(x, y) >= 0;
}

inline int operator<(SubString& x, SubString&  y) 
{
  return compare(x, y) < 0;
}

inline int operator<=(SubString& x, SubString&  y)
{
  return compare(x, y) <= 0;
}

inline int operator==(SubString& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(SubString& x, const char* t) 
{
  return compare(x, t) != 0;
}

inline int operator>(SubString& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(SubString& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(SubString& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(SubString& x, const char* t) 
{
  return compare(x, t) <= 0; 
}


inline SubString String::at(int first, int len)
{
  return SubString(this, first, len);
}

inline SubString String::before(int pos)
{
  return SubString(this, 0, pos);
}

inline SubString String::through(int pos)
{
  return SubString(this, 0, pos+1);
}

inline SubString String::after(int pos)
{
  return SubString(this, pos + 1, rep->len - (pos + 1));
}

inline SubString String::from(int pos)
{
  return SubString(this, pos, rep->len - pos);
}

inline int String::gsub(String& pat, String& r)
{
  return _gsub(pat.rep->s, pat.rep->len, r.rep->s, r.rep->len);
}

inline int String::gsub(SubString&  pat, String& r)
{
  return _gsub(&(pat.S->rep->s[pat.pos]), pat.len, r.rep->s, r.rep->len);
}

inline int String::gsub(Regex& pat, String& r)
{
  return _gsub(pat, r.rep->s, r.rep->len);
}

inline int String::gsub(const char* pat, String& r)
{
  return _gsub(pat, -1, r.rep->s, r.rep->len);
}

inline int String::gsub(const char* pat, const char* r)
{
  return _gsub(pat, -1, r, -1);
}

inline void String::del(String& y, int startpos = 0)
{
  del(search(startpos, rep->len, y.rep->s, y.rep->len), y.rep->len);
}

inline void String::del(SubString& y, int startpos = 0)
{
  del(search(startpos, rep->len, &(y.S->rep->s[y.pos]), y.len), y.len);
}

inline void String::del(char c, int startpos = 0)
{
  del(search(startpos, rep->len, c), 1);
}


inline Regex::Regex(String& x, int fast = 0, int bufsize = 40, 
             const char* transtable = 0)
{
  initialize(x.rep->s, x.rep->len, fast, bufsize, transtable);
}

inline Regex::Regex(const char* t, int fast = 0, int bufsize = 40, 
             const char* transtable = 0)
{
  initialize(t, -1, fast, bufsize, transtable);
}

  

//#endif

#endif
