/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

/* 
  String class implementation
 */

#include <String.h>
#include <std.h>
#include <regex.h>
#include <ctype.h>
#include "libconfig.h"

// error handling


void default_String_error_handler(char* msg)
{
  cerr << "Fatal String error. " << msg << "\n";
  exit(1);
}

one_arg_error_handler_t String_error_handler = default_String_error_handler;

one_arg_error_handler_t set_String_error_handler(one_arg_error_handler_t f)
{
  one_arg_error_handler_t old = String_error_handler;
  String_error_handler = f;
  return old;
}

void String::error(char* msg)
{
  (*String_error_handler)(msg);
}

//  globals

_Srep  _nil_Srep = { 0, 0, -1, { 0 } }; // nil strings point here
static String _nilString;               // nil SubStrings point here



/*
  allocation fcts  

  Because of the way they are used, these can't
  be done conveniently by overloading 'new'

  The padded version is called to try to allocate
  a reasonable amount of storage for any "dynamic-looking" string.
  Unpadded version used instead for original assignments & constructors
  to conserve space. 

  Both allocate a minimum of MINSIZE chars, where MINSIZE is close 
  to the smallest chunk that malloc will deal with given the _Srep overhead.
*/


#define PAD            2
#define MIN_SREP_SIZE  4
#define MAX_SREP_SIZE  (1 << (SHORTBITS - 1) - 1)

static int calc_size(int l, int pad)
{
  int siz;
  if (pad != 0)
    siz = l * PAD + MIN_SREP_SIZE;
  else if (l < MIN_SREP_SIZE)
    siz = MIN_SREP_SIZE;
  else 
    siz = l;
  if (siz > MAX_SREP_SIZE)
    (*String_error_handler)("Requested length out of range");
  return siz;
}

inline static _Srep*  new_Srep(int l, int pad = 0)
{
  int siz = calc_size(l, pad);
  unsigned allocsiz = sizeof(_Srep) + siz;
  _Srep* z = (_Srep *) (malloc(allocsiz));
  if (z == 0)
    (*String_error_handler)("Out of Memory");
    
  z->len = l;
  z->sz  = siz + 1;
  z->ref = 1;
  return z;
}

// expand a rep via realloc

inline static _Srep* expand_Srep(_Srep* x, int l, int pad = 0)
{
  int siz = calc_size(l, pad);
  unsigned allocsiz = sizeof(_Srep) + siz;

#ifdef SHOULD_FREE_TO_REALLOC
  free((void*)x);
#endif

  _Srep* z = (_Srep*)(realloc((void*)x, allocsiz));
  if (z == 0)
    (*String_error_handler)("Out of Memory");

  z->len = l;
  z->sz = siz + 1;
  z->ref = 1;
  return z;
}

/*
 the following inline fcts are specially designed to work
 in support of String classes, and are not meant as generic replacements
 for libc "str" functions.

 inline copy fcts -  I like left-to-right from->to arguments.
 all versions assume that `to' argument is non-null
*/

// copy n bytes
inline static void ncopy(const char* from, char* to, int n)
{
  if (n > 0 && from != to) bcopy((void*)from, (void*) to, n);
}

// copy n bytes, null-terminate
inline static void ncopy0(const char* from, char* to, int n)
{
  if (n > 0 && from != to) bcopy((void*)from, (void*) to, n);
  to[n] = 0;
}

// copy until null
inline static void scopy(const char* from, char* to)
{
  if (from != 0) while((*to++ = *from++) != 0);
}

// copy right-to-left
inline static void revcopy(const char* from, char* to, short n)
{
  if (from != 0) while (n-- > 0) *to-- = *from--;
}

inline static int slen(const char* t) // inline  strlen
{
  if (t == 0)
    return 0;
  else
  {
    const char* a = t;
    while (*a++ != 0);
    return a - 1 - t;
  }
}

// string compare: first argument is known to be non-null

inline static int scmp(const char* a, const char* b)
{
  if (b == 0)
    return *a != 0;
  else
  {
    signed char diff = 0;
    while ((diff = *a - *b++) == 0 && *a++ != 0);
    return diff;
  }
}

inline static int ncmp(const char* a, int al, const char* b, int bl)
{
  int n = al <? bl;
  signed char diff;
  while (n-- > 0) if ((diff = *a++ - *b++) != 0) return diff;
  return al - bl;
}

int fcompare(String& x, String& y)
{
  const char* a = x.rep->s;
  const char* b = y.rep->s;
  int al = x.rep->len;
  int bl = y.rep->len;
  int n = al <? bl;
  signed char diff = 0;
  while (n-- > 0)
  {
    char ac = *a++;
    char bc = *b++;
    if ((diff = ac - bc) != 0)
    {
      if (ac >= 'a' && ac <= 'z')
        ac = ac - 'a' + 'A';
      if (bc >= 'a' && bc <= 'z')
        bc = bc - 'a' + 'A';
      if ((diff = ac - bc) != 0)
        return diff;
    }
  }
  return al - bl;
}

int compare(String& x, const char* b)
{
  return scmp(x.rep->s, b);
}

int compare(String& x, String& y)
{
  return scmp(x.rep->s, y.rep->s);
}

int compare(String& x, SubString& y)
{
  return ncmp(x.rep->s, x.rep->len, &(y.S->rep->s[y.pos]), y.len);
}

int compare(SubString& x, String& y)
{
  return ncmp(&(x.S->rep->s[x.pos]), x.len, y.rep->s, y.rep->len);
}

int compare(SubString& x, SubString& y)
{
  return ncmp(&(x.S->rep->s[x.pos]), x.len, &(y.S->rep->s[y.pos]), y.len);
}

int compare(SubString& x, const char* b)
{
  if (b == 0)
    return x.len;
  else
  {
    const char* a = &(x.S->rep->s[x.pos]);
    int n = x.len;
    signed char diff;
    while (n-- > 0) if ((diff = *a++ - *b++) != 0) return diff;
    return (*b == 0) ? 0 : -1;
  }
}

/*
 index fcts
*/

int String::search(int start, int sl, char c)
{
  const char* s = rep->s;
  if (sl > 0)
  {
    if (start >= 0)
    {
      const char* a = &(s[start]);
      const char* lasta = &(s[sl]);
      while (a < lasta) if (*a++ == c) return --a - s;
    }
    else
    {
      const char* a = &(s[sl + start + 1]);
      while (--a >= s) if (*a == c) return a - s;
    }
  }
  return -1;
}

int String::search(int start, int sl, const char* t, int tl = -1)
{
  const char* s = rep->s;
  if (tl < 0)
    tl = slen(t);
  if (sl > 0 && tl > 0)
  {
    if (start >= 0)
    {
      const char* lasts = &(s[sl - tl]);
      const char* lastt = &(t[tl]);
      const char* p = &(s[start]);

      while (p <= lasts)
      {
        const char* x = p++;
        const char* y = t;
        while (*x++ == *y++) if (y >= lastt) return --p - s;
      }
    }
    else
    {
      const char* firsts = &(s[tl - 1]);
      const char* lastt =  &(t[tl - 1]);
      const char* p = &(s[sl + start + 1]); 

      while (--p >= firsts)
      {
        const char* x = p;
        const char* y = lastt;
        while (*x-- == *y--) if (y < t) return ++x - s;
      }
    }
  }
  return -1;
}


int String::match(int start, int sl, int exact, const char* t, int tl = -1)
{
  if (tl < 0)
    tl = slen(t);

  if (start < 0)
  {
    start = sl + start - tl + 1;
    if (start < 0 || (exact && start != 0))
      return 0;
  }
  else if (exact && sl - start != tl)
    return 0;

  if (sl == 0 || tl == 0 || sl - start < tl || start >= sl)
    return 0;

  int n = tl;
  const char* s = &(rep->s[start]);
  while (n-- > 0) if (*s++ != *t++) return 0;
  return tl;
}


/*
 * other utilities
 */

void String::copy(const char* news, int newlen = -1)
{
  if (newlen < 0)
    newlen = slen(news);

  if (newlen == 0)
  {
    if (rep->ref > 0 && --rep->ref == 0)
      delete rep;
    rep = &_nil_Srep;
    return;
  }
  else if (rep->ref != 1)
  {
    if (rep->ref > 0) --rep->ref;
    rep = new_Srep(newlen, PAD);
  }
  else if (newlen >= rep->sz)
    rep = expand_Srep(rep, newlen, PAD);
  else
    rep->len = newlen;

  ncopy0(news, rep->s, newlen);
}

void String::setlength(int newlen)
{
  if (rep->ref != 1)
  {
    if (rep->ref > 0) rep->ref--;
    _Srep* newrep = new_Srep(newlen, PAD);
    ncopy(rep->s, newrep->s, rep->len);
    rep = newrep;
  }
  else if (newlen > rep->sz)
    rep = expand_Srep(rep, newlen, PAD);
  else
    rep->len = newlen;
}

  
/* 
 concatenation:
 the different versions reflect different allocation, 
 copying & aliasing issues.
*/

void concat(String& x, String& y, String& r)
{
  int xl = x.rep->len;
  int yl = y.rep->len;
  int yrsame = y.rep == r.rep;
  r.setlength(xl+yl);
  ncopy(x.rep->s, r.rep->s, xl);
  char* ys = (yrsame)? r.rep->s : y.rep->s;
  ncopy0(ys, &(r.rep->s[xl]), yl);
}

void concat(SubString& x, SubString& y, String& r)
{
  int xl = x.len;
  int yl = y.len;
  int xrsame = x.S->rep == r.rep;
  int yrsame = y.S->rep == r.rep;
  if (xrsame || yrsame)
  {
    String s;
    s.setlength(xl+yl);
    ncopy(&(x.S->rep->s[x.pos]), s.rep->s, xl);
    ncopy0(&(y.S->rep->s[y.pos]), &(s.rep->s[xl]), yl);
    r = s;
  }
  else
  {
    r.setlength(xl+yl);
    ncopy(&(x.S->rep->s[x.pos]), r.rep->s, xl);
    ncopy0(&(y.S->rep->s[y.pos]), &(r.rep->s[xl]), yl);
  }
}

void concat(String& x, SubString& y, String& r)
{
  int xl = x.rep->len;
  int yl = y.len;
  int yrsame = y.S->rep == r.rep;
  if (yrsame)
  {
    String s;
    s.setlength(xl + yl);
    ncopy(x.rep->s, s.rep->s, xl);
    ncopy0(&(y.S->rep->s[y.pos]), &(s.rep->s[xl]), yl);
    r = s;
  }
  else
  {
    r.setlength(xl+yl);
    ncopy(x.rep->s, r.rep->s, xl);
    ncopy0(&(y.S->rep->s[y.pos]), &(r.rep->s[xl]), yl);
  }
}

void concat(SubString& x, String& y, String& r)
{
  int xl = x.len;
  int yl = y.rep->len;
  int xrsame = x.S->rep == r.rep;
  int yrsame = y.rep == r.rep;
  if (xrsame)
  {
    String s;
    s.setlength(xl + yl);
    ncopy(&(x.S->rep->s[x.pos]), s.rep->s, xl);
    ncopy0(y.rep->s, &(s.rep->s[xl]), yl);
    r = s;
  }
  else
  {
    r.setlength(xl+yl);
    ncopy(&(x.S->rep->s[x.pos]), r.rep->s, xl);
    char* ys = (yrsame)? r.rep->s : y.rep->s;
    ncopy0(ys, &(r.rep->s[xl]), yl);
  }
}

void concat(String& x, const char* y, String& r)
{
  int xl = x.rep->len;
  int yl = slen(y);
  r.setlength(xl+yl);
  ncopy(x.rep->s, r.rep->s, xl);
  ncopy0(y, &(r.rep->s[xl]), yl);
}

void concat(SubString& x, const char* y, String& r)
{
  int xl = x.len;
  int yl = slen(y);
  int xrsame = x.S->rep == r.rep;
  if (xrsame)
  {
    String s;
    s.setlength(xl+yl);
    ncopy(&(x.S->rep->s[x.pos]), s.rep->s, xl);
    ncopy0(y, &(s.rep->s[xl]), yl);
    r = s;
  }
  else
  {
    r.setlength(xl+yl);
    ncopy(&(x.S->rep->s[x.pos]), r.rep->s, xl);
    ncopy0(y, &(r.rep->s[xl]), yl);
  }
}

void concat(String& x, char y, String& r)
{
  int xl = x.rep->len;
  r.setlength(xl+1);
  ncopy(x.rep->s, r.rep->s, xl);
  r.rep->s[xl++] = y;
  r.rep->s[xl] = 0;
}


void concat(SubString& x, char y, String& r)
{
  int xl = x.len;
  int xrsame = x.S->rep == r.rep;
  if (xrsame)
  {
    String s;
    s.setlength(xl+1);
    ncopy(&(x.S->rep->s[x.pos]), s.rep->s, xl);
    s.rep->s[xl++] = y;
    s.rep->s[xl] = 0;
    r = s;
  }
  else
  {
    r.setlength(xl+1);
    ncopy(&(x.S->rep->s[x.pos]), r.rep->s, xl);
    r.rep->s[xl++] = y;
    r.rep->s[xl] = 0;
  }
}



void SubString::assign(_Srep* ysrc, const char* ys, int ylen=-1)
{
  _Srep* targ = S->rep;

  if (ylen < 0)
    ylen = slen(ys);

  if (len == 0 || pos >= targ->len)
    return;

  int sl = targ->len - len + ylen;

  if (targ->ref != 1)
  {
    if (targ->ref > 0) targ->ref--;
    _Srep* oldtarg = targ;
    targ = new_Srep(sl, PAD);
    ncopy(oldtarg->s, targ->s, pos);
    ncopy(ys, &(targ->s[pos]), ylen);
    scopy(&(oldtarg->s[pos + len]), &(targ->s[pos + ylen]));
  }
  else if (ysrc == targ || sl >= targ->sz)
  {
    _Srep* oldtarg = targ;
    targ = new_Srep(sl, PAD);
    ncopy(oldtarg->s, targ->s, pos);
    ncopy(ys, &(targ->s[pos]), ylen);
    scopy(&(oldtarg->s[pos + len]), &(targ->s[pos + ylen]));
    delete oldtarg;
  }
  else if (len == ylen)
    ncopy(ys, &(targ->s[pos]), len);
  else if (ylen < len)
  {
    ncopy(ys, &(targ->s[pos]), ylen);
    scopy(&(targ->s[pos + len]), &(targ->s[pos + ylen]));
    targ->len = sl;
  }
  else
  {
    revcopy(&(targ->s[targ->len]), &(targ->s[sl]), targ->len-pos-len +1);
    ncopy(ys, &(targ->s[pos]), ylen);
    targ->len = sl;
  }
  S->rep = targ;
}



SubString::SubString(String* x, int first, int l)
{
  if (first < 0 || l <= 0 || first + l > x->rep->len)
  {
    S = &_nilString; pos = len = 0;
  }
  else
  {
    S = x; pos = first; len = l;
  }
}


void Regex::initialize(const char* t, int tlen, int fast, int bufsize, 
                       const char* transtable)
{
  buf = new re_pattern_buffer;
  reg = new re_registers;
  if (fast)
    buf->fastmap = new char[256];
  else
    buf->fastmap = 0;
  buf->translate = (char*)transtable;
  if (tlen > bufsize)
    bufsize = tlen;
  buf->allocated = bufsize;
  buf->buffer = new char [buf->allocated];
  char* msg = re_compile_pattern((char*)t, tlen, buf);
  if (msg != 0)
    (*String_error_handler)(msg);
  else if (fast)
    re_compile_fastmap(buf);
}

Regex::Regex(const char* t, int fast = 0, int bufsize = 40, 
             const char* transtable = 0)
{
  initialize(t, slen(t), fast, bufsize, transtable);
}

Regex::Regex(String& x, int fast = 0, int bufsize = 40, 
             const char* transtable = 0)
{
  initialize(x.rep->s, x.rep->len, fast, bufsize, transtable);
}

Regex::~Regex()
{
  delete(buf->buffer);
  delete(buf->fastmap);
  delete(buf);
  delete(reg);
}

int Regex::search(const char* s, int len, int& matchlen, int startpos = 0)
{
  int matchpos, pos, range;
  if (startpos >= 0)
  {
    pos = startpos;
    range = len - startpos;
  }
  else
  {
    pos = len + startpos;
    range = -pos;
  }
  matchpos = re_search_2(buf, 0, 0, (char*)s, len, pos, range, reg, len);
  if (matchpos >= 0)
    matchlen = reg->end[0] - reg->start[0];
  else
    matchlen = 0;
  return matchpos;
}

int Regex::match(const char*s, int len, int p = 0)
{
  if (p < 0)
  {
    p += len;
    if (p >= len)
      return 0;
    return re_match_2(buf, 0, 0, (char*)s, p, 0, reg, p);
  }
  else if (p >= len)
    return 0;
  else
    return re_match_2(buf, 0, 0, (char*)s, len, p, reg, len);
}


/*
 * substring extraction 
 */


SubString String::at(int first, int len)
{
  return SubString(this, first, len);
}

SubString String::at(String& y, int startpos = 0)
{
  int first = search(startpos, rep->len, y.rep->s, y.rep->len);
  return SubString(this, first,  y.rep->len);
}

SubString String::before(String& y, int startpos = 0)
{
  int last = search(startpos, rep->len, y.rep->s, y.rep->len);
  return SubString(this, 0, last);
}

SubString String::after(String& y, int startpos = 0)
{
  int first = search(startpos, rep->len, y.rep->s, y.rep->len);
  return SubString(this, first + y.rep->len, rep->len - (first+y.rep->len));
}

SubString String::at(char c, int startpos = 0)
{
  int first = search(startpos, rep->len, c);
  return SubString(this, first, 1);
}

SubString String::before(char c, int startpos = 0)
{
  int last = search(startpos, rep->len, c);
  return SubString(this, 0, last);
}

SubString String::after(char c, int startpos = 0)
{
  int first = search(startpos, rep->len, c);
  return SubString(this, first + 1, rep->len - (first + 1));
}

SubString String::at(const char* t, int startpos = 0)
{
  int tlen = slen(t);
  int first = search(startpos, rep->len, t, tlen);
  return SubString(this, first, tlen);
}

SubString String::before(const char* t, int startpos = 0)
{
  int last = search(startpos, rep->len, t);
  return SubString(this, 0, last);
}

SubString String::after(const char* t, int startpos = 0)
{
  int tlen = slen(t);
  int first = search(startpos, rep->len, t, tlen);
  return SubString(this, first + tlen, rep->len - (first + tlen));
}


SubString String::at(SubString& y, int startpos = 0)
{
  int first = search(startpos, rep->len, &(y.S->rep->s[y.pos]), y.len);
  return SubString(this, first, y.len);
}

SubString String::before(SubString& y, int startpos = 0)
{
  int last = search(startpos, rep->len, &(y.S->rep->s[y.pos]), y.len);
  return SubString(this, 0, last);
}

SubString String::after(SubString& y, int startpos = 0)
{
  int first = search(startpos, rep->len, &(y.S->rep->s[y.pos]), y.len);
  return SubString(this, first + y.len, rep->len - (first + y.len));
}

SubString String::at(Regex& r, int startpos = 0)
{
  int first, mlen;
  first = r.search(rep->s, rep->len, mlen, startpos);
  return SubString(this, first, mlen);
}


SubString String::before(Regex& r, int startpos = 0)
{
  int first, mlen;
  first = r.search(rep->s, rep->len, mlen, startpos);
  return SubString(this, 0, first);
}

SubString String::after(Regex& r, int startpos = 0)
{
  int first, mlen;
  first = r.search(rep->s, rep->len, mlen, startpos);
  return SubString(this, first + mlen, rep->len - (first + mlen));
}

SubString String::before(int pos)
{
  return SubString(this, 0, pos);
}

SubString String::after(int pos)
{
  return SubString(this, pos + 1, rep->len - (pos + 1));
}


/*
 * substitution
 */


void String::_gsub(const char* pat, int pl, const char* r, int rl)
{
  int sl = rep->len;
  if (sl <= 0 || pl <= 0 || sl < pl)
    return;

  const char* s = rep->s;

  _Srep* nrep = new_Srep(2 * sl); // guess size
  char* x = nrep->s;

  int si = 0;
  int xi = 0;
  int remaining = sl;

  while (remaining >= pl)
  {
    int pos = search(si, sl, pat, pl);
    if (pos < 0)
      break;
    else
    {
      int mustfit = xi + remaining + rl - pl;
      if (mustfit >= nrep->sz)
      {
        nrep = expand_Srep(nrep, mustfit, PAD);
        x = nrep->s;
      }
      pos -= si;
      ncopy(&(s[si]), &(x[xi]), pos);
      ncopy(r, &(x[xi + pos]), rl);
      si += pos + pl;
      remaining -= pos + pl;
      xi += pos + rl;
    }
  }

  ncopy0(&(s[si]), &(x[xi]), remaining);
  nrep->len = xi + remaining;

  if (rep->ref == 1)
  {
    if (nrep->len <= rep->sz)   // fit back in if possible
    {
      rep->ref = 1;
      rep->len = nrep->len;
      ncopy0(nrep->s, rep->s, rep->len);
      delete(nrep);
      return;
    }
    else
      delete(rep);
  }
  else 
    if (rep->ref > 0) --rep->ref;

  rep = nrep;
}

void String::_gsub(Regex& pat, const char* r, int rl)
{
  int sl = rep->len;
  if (sl <= 0)
    return;

  const char* s = rep->s;

  _Srep* nrep = new_Srep(sl * 2, PAD); // guess len, readjust as necessary
  char* x = rep->s;

  int si = 0;
  int xi = 0;
  int remaining = sl;
  int  pos, pl;

  while (remaining > 0)
  {
    pos = pat.search(s, sl, pl, si);
    if (pos < 0 || pl <= 0)
      break;
    else
    {
      int mustfit = xi + remaining + rl - pl;
      if (mustfit >= nrep->sz)
      {
        nrep = expand_Srep(nrep, mustfit, PAD);
        x = nrep->s;
      }
      ncopy(&(s[si]), &(x[xi]), pos);
      ncopy(r, &(x[xi + pos]), rl);
      si += pos + pl;
      remaining -= pos + pl;
      xi += pos + rl;
    }
  }

  ncopy0(&(s[si]), &(x[xi]), remaining);
  nrep->len = xi + remaining;

  if (rep->ref == 1)
  {
    if (nrep->len <= rep->sz)   // fit back in if possible
    {
      rep->ref = 1;
      rep->len = nrep->len;
      ncopy0(nrep->s, rep->s, rep->len);
      delete(nrep);
      return;
    }
    else
      delete(rep);
  }
  else
    if (rep->ref > 0) --rep->ref;

  rep = nrep;
}


void String::gsub(String& pat, String& r)
{
  _gsub(pat.rep->s, pat.rep->len, r.rep->s, r.rep->len);
}

void String::gsub(SubString&  pat, String& r)
{
  _gsub(&(pat.S->rep->s[pat.pos]), pat.len, r.rep->s, r.rep->len);
}

void String::gsub(Regex& pat, String& r)
{
  _gsub(pat, r.rep->s, r.rep->len);
}

void String::gsub(const char* pat, String& r)
{
  _gsub(pat, slen(pat), r.rep->s, r.rep->len);
}

void String::gsub(const char* pat, const char* r)
{
  _gsub(pat, slen(pat), r, slen(r));
}


/*
 * deletion
 */

void String::del(int pos, int len)
{
  if (pos <= 0 || rep->len <= 0 || pos + len > rep->len)
    return;
  
  _Srep* nrep = rep;
  int nlen = rep->len - len;
  if (rep->ref != 1)
  {
    if (rep->ref > 0) rep->ref--;
    nrep = new_Srep(nlen, PAD);
    ncopy(rep->s, nrep->s, pos);
  }
  scopy(&(rep->s[pos + len]), &(nrep->s[pos]));
  nrep->len = nlen;
  rep = nrep;
}

void String::del(String& y, int startpos = 0)
{
  del(search(startpos, rep->len, y.rep->s, y.rep->len), y.rep->len);
}

void String::del(SubString& y, int startpos = 0)
{
  del(search(startpos, rep->len, &(y.S->rep->s[y.pos]), y.len), y.len);
}

void String::del(const char* t, int startpos = 0)
{
  int tlen = slen(t);
  int p = search(startpos, rep->len, t, tlen);
  del(p, tlen);
}

void String::del(char c, int startpos = 0)
{
  del(search(startpos, rep->len, c), 1);
}

void String::del(Regex& r, int startpos = 0)
{
  int first, mlen;
  first = r.search(rep->s, rep->len, mlen, startpos);
  del(first, mlen);
}

/*
 decomposition
 */

int decompose(String& x, String& l, String& m, String& r,
               Regex& pat, int startpos = 0)
{
  int p, patlen;
  p = pat.search(x.rep->s, x.rep->len, patlen, startpos);
  if (p >= 0 && patlen >= 0 && patlen <= x.rep->len)
  {
    String tmp = x;             // in case x = l/m/r
    int sl = tmp.rep->len;
    l.copy(tmp.rep->s, p);
    m.copy(&(tmp.rep->s[p]), patlen);
    r.copy(&(tmp.rep->s[p + patlen]), sl - (p + patlen));
    return 1;
  }
  else
    return 0;
}

/*
 * split/join
 */


int split(String& src, String results[], int n, String& sep)
{
  String x = src;
  _Srep* xs = x.rep;
  const char* s = xs->s;
  int sl = xs->len;
  int i = 0;
  int pos = 0;
  while (i < n && pos < sl)
  {
    int p = x.search(pos, sl, sep.rep->s, sep.rep->len);
    if (p < 0)
      p = sl;
    results[i].copy(&(s[pos]), p - pos);
    i++;
    pos = p + sep.rep->len;
  }
  return(i);
}

int split(String& src, String results[], int n, Regex& r)
{
  String x = src;
  char* s = x.rep->s;
  int sl = x.rep->len;
  int i = 0;
  int pos = 0;
  int p, matchlen;
  while (i < n && pos < sl)
  {
    p = r.search(s, sl, matchlen, pos);
    if (p < 0)
      p = sl;
    results[i].copy(&(s[pos]), p - pos);
    i++;
    pos = p + matchlen;
  }
  return(i);
}


String join(String src[], int n, String& separator)
{
  String sep = separator;
  int xlen = 0;
  for (int i = 0; i < n; ++i)
    xlen += src[i].rep->len;
  xlen += (n - 1) * sep.rep->len;

  String x;
  x.rep = new_Srep(xlen);

  int j = 0;
  
  for (i = 0; i < n - 1; ++i)
  {
    ncopy(src[i].rep->s, &(x.rep->s[j]), src[i].rep->len);
    j += src[i].rep->len;
    ncopy(sep.rep->s, &(x.rep->s[j]), sep.rep->len);
    j += sep.rep->len;
  }
  ncopy0(src[i].rep->s, &(x.rep->s[j]), src[i].rep->len);
  return x;
}

  
/*
 misc
*/

    
String reverse(String& x)
{
  String y;
  y.copy(x.rep->s, x.rep->len);
  if (y.rep->len > 0)
  {
    char* a = y.rep->s;
    char* b = &(y.rep->s[y.rep->len - 1]);
    while (a < b)
    {
      char t = *a;
      *a++ = *b;
      *b-- = t;
    }
  }
  return y;
}


String replicate(char c, int n)
{
  String w;
  w.rep = new_Srep(n);
  char* p = w.rep->s;
  while (n-- > 0)
    *p++ = c;
  *p = 0;
  return w;
}

String replicate(String& y, int n)
{
  String w;
  w.rep = new_Srep(n * y.rep->len);
  char* p = w.rep->s;
  while (n-- > 0)
  {
    ncopy(y.rep->s, p, y.rep->len);
    p += y.rep->len;
  }
  *p = 0;
  return w;
}

String upcase(String& y)
{
  String x;
  x.copy(y.rep->s, y.rep->len);
  char* e = &(x.rep->s[x.rep->len]);
  char* p = x.rep->s;
  for (; p < e; ++p)
    if (*p >= 'a' && *p <= 'z') *p = *p - 'a' + 'A';
  return x;
}

String downcase(String& y)
{
  String x;
  x.copy(y.rep->s, y.rep->len);
  char* e = &(x.rep->s[x.rep->len]);
  char* p = x.rep->s;
  for (; p < e; ++p)
    if (*p >= 'A' && *p <= 'Z') *p = *p - 'A' + 'a';
  return x;
}

String capitalize(String& y)
{
  String x;
  x.copy(y.rep->s, y.rep->len);
  char* e = &(x.rep->s[x.rep->len]);
  for (char* p = x.rep->s; p < e; ++p)
  {
    int at_word;
    if (at_word = *p >= 'a' && *p <= 'z')
      *p = *p - 'a' + 'A';
    else 
      at_word = (*p >= 'A' && *p <= 'Z') || (*p >= '0' && *p <= '9');

    if (at_word)
    {
      while (++p < e)
      {
        if (*p >= 'A' && *p <= 'Z')
          *p = *p - 'A' + 'a';
        else if (!((*p >= 'a' && *p <= 'z') || (*p >= '0' && *p <= '9')))
          break;
      }
    }
  }
  return x;
}

String common_prefix(String& x, String& y, int startpos = 0)
{
  const char* xs = &(x.rep->s[startpos]);
  const char* ss = xs;
  const char* topx = &(x.rep->s[x.rep->len]);
  const char* ys = &(y.rep->s[startpos]);
  const char* topy = &(y.rep->s[y.rep->len]);
  for (int l = 0; xs < topx && ys < topy && *xs++ == *ys++; ++l);
  return String(ss, l);
}


String common_suffix(String& x, String& y, int startpos = -1)
{
  const char* xs = &(x.rep->s[x.rep->len + startpos]);
  const char* botx = x.rep->s;
  const char* ys = &(y.rep->s[y.rep->len + startpos]);
  const char* boty = y.rep->s;
  for (int l = 0; xs >= botx && ys >= boty && *xs == *ys ; --xs, --ys, ++l);
  return String(++xs, l);
}

// IO

istream& operator>>(istream& s, String& x)
{
  char ch;
  int i = 0;
  String w;
  w.rep = new_Srep(20);        // guess length;
  s >> WS;
  while (s.good())
  {
    s.get(ch);
    if (isspace(ch))
      break;
    if (i >= w.rep->sz - 1)
      w.rep = expand_Srep(w.rep, i+1, PAD);
    w.rep->s[i++] = ch;
  }
  w.rep->s[i] = 0;
  w.rep->len = i;
  x = w;
  s.failif(i == 0);
  return s;
}


ostream& operator<<(ostream& s, SubString& x)
{ 
  const char* a = &(x.S->rep->s[x.pos]);
  const char* lasta = &(a[x.len]);
  while (a < lasta)
    s.put(*a++);
  return(s);
}


/*
 some built-in Regular expressions
*/

Regex RXwhite("[ \n\t]+", 1);
Regex RXint("-?[0-9]+", 1);
Regex RXdouble("-?\\(\\([0-9]+\\.[0-9]*\\)\\|\\([0-9]+\\)\\|\\(\\.[0-9]+\\)\\)\\([eE][---+]?[0-9]+\\)?", 1, 200);
Regex RXalpha("[A-Za-z]+", 1);
Regex RXlowercase("[a-z]+", 1);
Regex RXuppercase("[A-Z]+", 1);
Regex RXalphanum("[0-9A-Za-z]+", 1);
Regex RXidentifier("[A-Za-z_][A-Za-z0-9_]*", 1);
