/* Functions to write to uart on a print command.
 *
 * Copyright (c) 2004, 2005, 2006, 2007 Cavium Networks.
 *
 * The authors hereby grant permission to use, copy, modify, distribute,
 * and license this software and its documentation for any purpose, provided
 * that existing copyright notices are retained in all copies and that this
 * notice is included verbatim in any distributions. No written agreement,
 * license, or royalty fee is required for any of the authorized uses.
 * Modifications to this software may be copyrighted by their authors
 * and need not follow the licensing terms described here, provided that
 * the new terms are clearly indicated on the first page of each file where
 * they apply.
 */

#include <unistd.h>

#include "octeon-uart.h"

CVMX_SHARED static cvmx_spinlock_t octeon_uart_lock =
  { CVMX_SPINLOCK_UNLOCKED_VAL };

/* Put a single byte to uart port. UART_INDEX is the uart to write to (0/1).
   CH contains the byte to write.  */
static void 
octeon_uart_write_byte (int uart_index, uint8_t ch)
{
  cvmx_uart_lsr_t lsrval;

  /* Spin until there is room */
  do
    {
      lsrval.u64 = cvmx_read_csr (CVMX_MIO_UARTX_LSR (uart_index));
    } while (lsrval.s.thre == 0);

  /* Write the byte */
  cvmx_write_csr (CVMX_MIO_UARTX_THR (uart_index), ch);
}

/* Write out the PP banner without using any C library functions to uart
   specified by UART_INDEX.  */
static void 
octeon_uart_write_banner (int uart_index)
{
  const uint64_t coreid = cvmx_get_core_num ();

  octeon_uart_write_byte (uart_index, 'P');
  octeon_uart_write_byte (uart_index, 'P');
  if (coreid < 10)
    octeon_uart_write_byte (uart_index, coreid + '0');
  else
    {
      octeon_uart_write_byte (uart_index, '1');
      octeon_uart_write_byte (uart_index, coreid - 10 + '0');
    }
  octeon_uart_write_byte (uart_index, ':');
  octeon_uart_write_byte (uart_index, '~');
  octeon_uart_write_byte (uart_index, 'C');
  octeon_uart_write_byte (uart_index, 'O');
  octeon_uart_write_byte (uart_index, 'N');
  octeon_uart_write_byte (uart_index, 'S');
  octeon_uart_write_byte (uart_index, 'O');
  octeon_uart_write_byte (uart_index, 'L');
  octeon_uart_write_byte (uart_index, 'E');
  octeon_uart_write_byte (uart_index, '-');
  octeon_uart_write_byte (uart_index, '>');
  octeon_uart_write_byte (uart_index, ' ');
}

void
__octeon_uart_lock (void)
{
  cvmx_spinlock_lock (&octeon_uart_lock);
}

unsigned
__octeon_uart_trylock (void)
{
  return cvmx_spinlock_trylock (&octeon_uart_lock);
}

void
__octeon_uart_unlock (void)
{
  cvmx_spinlock_unlock (&octeon_uart_lock);
}

/* Write bytes to the uart specified by UART_INDEX into BUFFER of bytes 
   in LEN.  Don't transform the original text and do not lock.  */
int 
__octeon_uart_write_raw (int uart_index, const char *buffer, size_t len)
{
  size_t ret = len;

  while (len--)
    octeon_uart_write_byte (uart_index, *buffer++);

  return ret;
}

/* Write bytes to the uart specified by UART_INDEX into BUFFER of bytes 
   in BUFFER_LEN.  */
int 
__octeon_uart_write (int uart_index, const char *buffer, size_t buffer_len)
{
  cvmx_spinlock_lock (&octeon_uart_lock);
  octeon_uart_write_banner (uart_index);

  /* Just loop writing one byte at a time */
  while (buffer_len)
    {
      if (*buffer == '\n')
        octeon_uart_write_byte (uart_index, '\r');
      octeon_uart_write_byte (uart_index, *buffer);
      buffer++;
      buffer_len--;
    }

  cvmx_spinlock_unlock (&octeon_uart_lock);

  return buffer_len;
}
