#ifndef CYGONCE_PTHREAD_H
#define CYGONCE_PTHREAD_H
//=============================================================================
//
//      pthread.h
//
//      POSIX pthread header
//
//=============================================================================
//####COPYRIGHTBEGIN####
//                                                                          
// -------------------------------------------                              
// The contents of this file are subject to the Red Hat eCos Public License 
// Version 1.1 (the "License"); you may not use this file except in         
// compliance with the License.  You may obtain a copy of the License at    
// http://www.redhat.com/                                                   
//                                                                          
// Software distributed under the License is distributed on an "AS IS"      
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the 
// License for the specific language governing rights and limitations under 
// the License.                                                             
//                                                                          
// The Original Code is eCos - Embedded Configurable Operating System,      
// released September 30, 1998.                                             
//                                                                          
// The Initial Developer of the Original Code is Red Hat.                   
// Portions created by Red Hat are                                          
// Copyright (C) 1998, 1999, 2000 Red Hat, Inc.                             
// All Rights Reserved.                                                     
// -------------------------------------------                              
//                                                                          
//####COPYRIGHTEND####
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     nickg
// Contributors:  nickg
// Date:          2000-03-17
// Purpose:       POSIX pthread header
// Description:   This header contains all the definitions needed to support
//                pthreads under eCos. The reader is referred to the POSIX
//                standard or equivalent documentation for details of the
//                functionality contained herein.
//              
// Usage:
//              #include <pthread.h>
//              ...
//              
//
//####DESCRIPTIONEND####
//
//=============================================================================

#include <pkgconf/hal.h>
#include <pkgconf/kernel.h>
#include <pkgconf/posix.h>

#include <cyg/infra/cyg_type.h>

#include <cyg/hal/hal_arch.h>   // CYGNUM_HAL_STACK_SIZE_MINIMUM

#include <stddef.h>             // NULL, size_t

#include <limits.h>

#include <sys/types.h>

#include <sched.h>              // SCHED_*

//=============================================================================
// General thread operations

//-----------------------------------------------------------------------------
// Thread creation and management.

// Create a thread.
externC int pthread_create ( pthread_t *thread,
                             const pthread_attr_t *attr,
                             void *(*start_routine) (void *),
                             void *arg);

// Get current thread id.
externC pthread_t pthread_self ( void );

// Compare two thread identifiers.
externC int pthread_equal (pthread_t thread1, pthread_t thread2);

// Terminate current thread.
externC void pthread_exit (void *retval) CYGBLD_ATTRIB_NORET;

// Wait for the thread to terminate. If thread_return is not NULL then
// the retval from the thread's call to pthread_exit() is stored at
// *thread_return.
externC int pthread_join (pthread_t thread, void **thread_return);

// Set the detachstate of the thread to "detached". The thread then does not
// need to be joined and its resources will be freed when it exits.
externC int pthread_detach (pthread_t thread);

//-----------------------------------------------------------------------------
// Thread attribute handling.

// Initialize attributes object with default attributes:
// detachstate          == PTHREAD_JOINABLE
// scope                == PTHREAD_SCOPE_SYSTEM
// inheritsched         == PTHREAD_EXPLICIT_SCHED
// schedpolicy          == SCHED_OTHER
// schedparam           == unset
// stackaddr            == unset
// stacksize            == 0
// 
externC int pthread_attr_init (pthread_attr_t *attr);

// Destroy thread attributes object
externC int pthread_attr_destroy (pthread_attr_t *attr);


// Set the detachstate attribute
externC int pthread_attr_setdetachstate (pthread_attr_t *attr,
                                         int detachstate);

// Get the detachstate attribute
externC int pthread_attr_getdetachstate (const pthread_attr_t *attr,
                                         int *detachstate);


// Set scheduling contention scope
externC int pthread_attr_setscope (pthread_attr_t *attr, int scope);

// Get scheduling contention scope
externC int pthread_attr_getscope (const pthread_attr_t *attr, int *scope);


// Set scheduling inheritance attribute
externC int pthread_attr_setinheritsched (pthread_attr_t *attr, int inherit);

// Get scheduling inheritance attribute
externC int pthread_attr_getinheritsched (const pthread_attr_t *attr,
                                          int *inherit);


// Set scheduling policy
externC int pthread_attr_setschedpolicy (pthread_attr_t *attr, int policy);

// Get scheduling policy
externC int pthread_attr_getschedpolicy (const pthread_attr_t *attr,
                                         int *policy);


// Set scheduling parameters
externC int pthread_attr_setschedparam (pthread_attr_t *attr,
				        const struct sched_param *param);

// Get scheduling parameters
externC int pthread_attr_getschedparam (const pthread_attr_t *attr,
                                        struct sched_param *param);


// Set starting address of stack. Whether this is at the start or end of
// the memory block allocated for the stack depends on whether the stack
// grows up or down.
externC int pthread_attr_setstackaddr (pthread_attr_t *attr, void *stackaddr);

// Get any previously set stack address.
externC int pthread_attr_getstackaddr (const pthread_attr_t *attr,
                                       void **stackaddr);


// Set minimum creation stack size.
externC int pthread_attr_setstacksize (pthread_attr_t *attr,
                                       size_t stacksize);

// Get current minimal stack size.
externC int pthread_attr_getstacksize (const pthread_attr_t *attr,
                                       size_t *stacksize);

//-----------------------------------------------------------------------------
// Thread scheduling controls

// Set scheduling policy and parameters for the thread
externC int pthread_setschedparam (pthread_t thread,
                                   int policy,
                                   const struct sched_param *param);

// Get scheduling policy and parameters for the thread
externC int pthread_getschedparam (pthread_t thread,
                                   int *policy,
                                   struct sched_param *param);



//=============================================================================
// Mutexes

//-----------------------------------------------------------------------------
// Mutex attributes manipulation functions

// Initialize attribute object
externC int pthread_mutexattr_init ( pthread_mutexattr_t *attr);

// Destroy attribute object
externC int pthread_mutexattr_destroy ( pthread_mutexattr_t *attr);

#if defined(_POSIX_THREAD_PRIO_INHERIT) || defined(_POSIX_THREAD_PRIO_PROTECT)

// Set priority inversion protection protocol
externC int pthread_mutexattr_setprotocol ( pthread_mutexattr_t *attr,
                                            int protocol);

// Get priority inversion protection protocol
externC int pthread_mutexattr_getprotocol ( pthread_mutexattr_t *attr,
                                            int *protocol);

#if defined(_POSIX_THREAD_PRIO_PROTECT)

// Set priority for priority ceiling protocol
externC int pthread_mutexattr_setprioceiling ( pthread_mutexattr_t *attr,
                                               int prioceiling);

// Get priority for priority ceiling protocol
externC int pthread_mutexattr_getprioceiling ( pthread_mutexattr_t *attr,
                                               int *prioceiling);

// Set priority ceiling of given thread, returning old ceiling.
externC int pthread_mutex_setprioceiling( pthread_mutex_t *mutex,
                                          int prioceiling,
                                          int *old_ceiling);

// Get priority ceiling of given thread
externC int pthread_mutex_getprioceiling( pthread_mutex_t *mutex,
                                          int *prioceiling);
#endif

#endif

//-----------------------------------------------------------------------------
// Mutex functions

// Initialize mutex. If mutex_attr is NULL, use default attributes.
externC int pthread_mutex_init (pthread_mutex_t *mutex,
                                const pthread_mutexattr_t *mutex_attr);

// Destroy mutex.
externC int pthread_mutex_destroy (pthread_mutex_t *mutex);

// Lock mutex, waiting for it if necessary.
externC int pthread_mutex_lock (pthread_mutex_t *mutex);

// Try to lock mutex.
externC int pthread_mutex_trylock (pthread_mutex_t *mutex);


// Unlock mutex.
externC int pthread_mutex_unlock (pthread_mutex_t *mutex);



//=============================================================================
// Condition Variables

//-----------------------------------------------------------------------------
// Attribute manipulation functions
// We do not actually support any attributes at present, so these do nothing.

// Initialize condition variable attributes
externC int pthread_condattr_init (pthread_condattr_t *attr);

// Destroy condition variable attributes
externC int pthread_condattr_destroy (pthread_condattr_t *attr);

//-----------------------------------------------------------------------------
// Condition variable functions

// Initialize condition variable.
externC int pthread_cond_init (pthread_cond_t *cond,
                               const pthread_condattr_t *attr);

// Destroy condition variable.
externC int pthread_cond_destroy (pthread_cond_t *cond);

// Wake up one thread waiting for condition variable
externC int pthread_cond_signal (pthread_cond_t *cond);

// Wake up all threads waiting for condition variable
externC int pthread_cond_broadcast (pthread_cond_t *cond);

// Block on condition variable until signalled. The mutex is
// assumed to be locked before this call, will be unlocked
// during the wait, and will be re-locked on wakeup.
externC int pthread_cond_wait (pthread_cond_t *cond,
                               pthread_mutex_t *mutex);

// Block on condition variable until signalled, or the timeout expires.
externC int pthread_cond_timedwait (pthread_cond_t *cond,
                                    pthread_mutex_t *mutex,
                                    const struct timespec *abstime);



//=============================================================================
// Dynamic package initialization

// Initializer for pthread_once_t instances
#define PTHREAD_ONCE_INIT       0

// Call init_routine just the once per control variable.
externC int pthread_once (pthread_once_t *once_control,
                          void (*init_routine) (void));



//=============================================================================
//Thread specific data

// Create a key to identify a location in the thread specific data area.
// Each thread has its own distinct thread-specific data area but all are
// addressed by the same keys. The destructor function is called whenever a
// thread exits and the value associated with the key is non-NULL.
externC int pthread_key_create (pthread_key_t *key,
                                void (*destructor) (void *));

// Delete key.
externC int pthread_key_delete (pthread_key_t key);

// Store the pointer value in the thread-specific data slot addressed
// by the key.
externC int pthread_setspecific (pthread_key_t key, const void *pointer);

// Retrieve the pointer value in the thread-specific data slot addressed
// by the key.
externC void *pthread_getspecific (pthread_key_t key);



//=============================================================================
// Thread Cancellation

//-----------------------------------------------------------------------------
// Data structure used to manage cleanup functions

struct pthread_cleanup_buffer
{
    struct pthread_cleanup_buffer *prev;        // Chain cleanup buffers
    void (*routine) (void *);     	        // Function to call
    void *arg;				        // Arg to pass
};

//-----------------------------------------------------------------------------
// Thread cancelled return value.
// This is a value returned as the retval in pthread_join() of a
// thread that has been cancelled. By making it the address of a
// location we define we can ensure that it differs from NULL and any
// other valid pointer (as required by the standard).

externC int pthread_canceled;

#define PTHREAD_CANCELED                ((void *)(&pthread_canceled))

//-----------------------------------------------------------------------------
// Cancelability enable and type

#define PTHREAD_CANCEL_ENABLE           1
#define PTHREAD_CANCEL_DISABLE          2

#define PTHREAD_CANCEL_ASYNCHRONOUS     1
#define PTHREAD_CANCEL_DEFERRED         2

//-----------------------------------------------------------------------------
// Functions

// Set cancel state of current thread to ENABLE or DISABLE.
// Returns old state in *oldstate.
externC int pthread_setcancelstate (int state, int *oldstate);

// Set cancel type of current thread to ASYNCHRONOUS or DEFERRED.
// Returns old type in *oldtype.
externC int pthread_setcanceltype (int type, int *oldtype);

// Cancel the thread.
externC int pthread_cancel (pthread_t thread);

// Test for a pending cancellation for the current thread and terminate
// the thread if there is one.
externC void pthread_testcancel (void);


// Install a cleanup routine.
// Note that pthread_cleanup_push() and pthread_cleanup_pop() are macros that
// must be used in matching pairs and at the same brace nesting level.
#define pthread_cleanup_push(routine,arg)                       \
    {                                                           \
        struct pthread_cleanup_buffer _buffer_;                 \
        pthread_cleanup_push_inner (&_buffer_, (routine), (arg));

// Remove a cleanup handler installed by the matching pthread_cleanup_push().
// If execute is non-zero, the handler function is called.
#define pthread_cleanup_pop(execute)                            \
        pthread_cleanup_pop_inner (&_buffer_, (execute));       \
    }


// These two functions actually implement the cleanup push and pop functionality.
externC void pthread_cleanup_push_inner (struct pthread_cleanup_buffer *buffer,
                                         void (*routine) (void *),
                                         void *arg);

externC void pthread_cleanup_pop_inner (struct pthread_cleanup_buffer *buffer,
                                        int execute);


//-----------------------------------------------------------------------------
#endif // ifndef CYGONCE_PTHREAD_H
// End of pthread.h
