/*
 * Copyright (C) 2005 Mike Christie, All rights reserved.
 *
 * This file is released under the GPL.
 *
 * Basic, very basic, support for HP StorageWorks and FSC FibreCat
 */
#include <scsi/scsi.h>
#include <scsi/scsi_cmnd.h>

#include "dm.h"
#include "dm-hw-handler.h"

struct hp_sw_handler {
	unsigned char sense[SCSI_SENSE_BUFFERSIZE];
};

static void hp_sw_endio(struct request *rq)
{
	struct path *path = rq->end_io_data;

	/*
	 * TODO: something.. we have the sense and scsi bytes
	 */
	if (rq->errors) {
		DMINFO("hp_sw: START_STOP on %s completed %d",
			path->dev->name, rq->errors);
	}

	dm_pg_init_complete(path, 0);
	__blk_put_request(rq->q, rq);
}

static struct request *hp_sw_get_request(struct hp_sw_handler *h,
					struct path *path)
{
	struct request *rq;
	struct block_device *bdev = path->dev->bdev;
	struct request_queue *q = bdev_get_queue(bdev);

	rq = blk_get_request(q, READ, __GFP_WAIT);
	if (!rq)
		return NULL;

	rq->end_io = hp_sw_endio;
	rq->end_io_data = path;
	rq->rq_disk = bdev->bd_contains->bd_disk;
	rq->sense = h->sense;
	memset(rq->sense, 0, SCSI_SENSE_BUFFERSIZE);
	rq->sense_len = 0;
	/*
	 * TODO: make me configurable
	 */
	rq->timeout = 30;
	rq->flags |= (REQ_BLOCK_PC | REQ_FAILFAST | REQ_NOMERGE);

	memset(&rq->cmd, 0, BLK_MAX_CDB);
	rq->cmd[0] = START_STOP;
	/* Start spin cycle */
	rq->cmd[4] = 1;
	rq->cmd_len = COMMAND_SIZE(rq->cmd[0]);

	return rq;
}

static void hp_sw_pg_init(struct hw_handler *hwh, unsigned bypassed,
			   struct path *path)
{
	struct request *rq;
	struct request_queue *q = bdev_get_queue(path->dev->bdev);

	/*
	 * We can either blindly init the pg (then look at the sense),
	 * or we can send some commands to get the state here (then
	 * possibly send the fo cmnd), or we can also have the
	 * initial state passed into us and then get an update here.
	 */
	if (!q) {
		DMERR("hp_sw: no queue!");
		goto fail;
	}

	if (blk_get_queue(q))
		goto fail;

	rq = hp_sw_get_request(hwh->context, path);
	if (!rq) {
		DMERR("hp_sw: could not allocate request for START_STOP");
		goto rel_queue;
	}

	DMINFO("hp_sw: queueing START_STOP command on %s",
		path->dev->name);
	elv_add_request(q, rq, ELEVATOR_INSERT_FRONT, 1);
	blk_put_queue(q);
	return;

rel_queue:
	blk_put_queue(q);
fail:
	dm_pg_init_complete(path, 0);
}

static int hp_sw_create(struct hw_handler *hwh, unsigned argc, char **argv)
{
	struct hp_sw_handler *h;

	h = kmalloc(sizeof(*h), GFP_KERNEL);
	if (!h) {
		DMERR("hp_sw: could not allocate hw_handler");
		return -ENOMEM;
	}

	hwh->context = h;
	return 0;
}

static void hp_sw_destroy(struct hw_handler *hwh)
{
	struct hp_sw_handler *h = hwh->context;

	kfree(h);
	hwh->context = NULL;
}

static struct hw_handler_type hp_sw_hwh = {
	.name = "hp_sw",
	.module = THIS_MODULE,
	.create = hp_sw_create,
	.destroy = hp_sw_destroy,
	.pg_init = hp_sw_pg_init,
};

static int __init hp_sw_init(void)
{
	int r;

	r = dm_register_hw_handler(&hp_sw_hwh);
	if (r < 0)
		DMERR("hp_sw: register failed %d", r);

	DMINFO("hp_sw version 0.4 loaded");

	return r;
}

static void __exit hp_sw_exit(void)
{
	int r;

	r = dm_unregister_hw_handler(&hp_sw_hwh);
	if (r < 0)
		DMERR("hp_sw: unregister failed %d", r);
}

module_init(hp_sw_init);
module_exit(hp_sw_exit);

MODULE_DESCRIPTION("HP StorageWorks and FSC FibreCat support for dm-multipath");
MODULE_AUTHOR("Mike Christie <michaelc@cs.wisc.edu>");
MODULE_LICENSE("GPL");
